import pns


def get_problem():
    problem = pns.Problem()
    problem.set('file_name', 'Bad_Zell_Case_Study')
    problem.set('mass_unit', 'unit')
    problem.set('time_unit', 'y')
    problem.set('money_unit', 'EUR')

    max_identical_fermenters = 2
    max_identical_chp = 3
    supplier_count = 8
    suppliers = [f'L{i + 1}' for i in range(supplier_count)]
    # individual_suppliers = [x for x in 'ABCDEFGHIJKLMNPQRST']
    # individual_supplier_group = {
    #     'A': 'L1',
    #     'B': 'L2',
    #     'C': 'L3',
    #     'D': 'L3',
    #     'E': 'L4',
    #     'F': 'L4',
    #     'G': 'L4',
    #     'H': 'L5',
    #     'I': 'L5',
    #     'J': 'L6',
    #     'K': 'L6',
    #     'L': 'L3',
    #     'M': 'L7',
    #     'N': 'L7',
    #     'P': 'L8',
    #     'Q': 'L8',
    #     'R': 'L2',
    #     'S': 'L8',
    #     'T': 'L8',
    # }
    location_count = 3
    locations = [f'Loc{i + 1}' for i in range(location_count)]
    chp_locations = locations.copy()
    chp_locations.append('CENTER')
    distance = {  # km
        'L1': {'Loc1': 1.6, 'Loc2': 3.4, 'Loc3': 0.0},
        'L2': {'Loc1': 3.3, 'Loc2': 4.7, 'Loc3': 4.0},
        'L3': {'Loc1': 2.7, 'Loc2': 4.6, 'Loc3': 1.2},
        'L4': {'Loc1': 1.9, 'Loc2': 1.4, 'Loc3': 3.3},
        'L5': {'Loc1': 0.3, 'Loc2': 2.1, 'Loc3': 2.1},
        'L6': {'Loc1': 1.5, 'Loc2': 2.9, 'Loc3': 3.0},
        'L7': {'Loc1': 3.1, 'Loc2': 3.0, 'Loc3': 2.4},
        'L8': {'Loc1': 3.8, 'Loc2': 1.9, 'Loc3': 3.7},
    }
    pipe_sections = ['Pipe1', 'Pipe2', 'Pipe3to1']
    pipe_section_length = {'Pipe1': 1.1, 'Pipe2': 1.0, 'Pipe3to1': 1.3}  # km
    pipe_requirements = {
        'Loc1': ['Pipe1'],
        'Loc2': ['Pipe2'],
        'Loc3': ['Pipe1', 'Pipe3to1'],
    }
    substrates = ['Manure', 'Inter', 'Grass', 'Corn']
    substrate_data = {
        'AS_FM': {'Manure': True, 'Inter': False, 'Grass': False, 'Corn': False},
        'DM': {'Manure': 0.09, 'Inter': 0.24, 'Grass': 0.30, 'Corn': 0.33},  # ratio
        'CH4': {'Manure': 200, 'Inter': 300, 'Grass': 300, 'Corn': 340},  # m3/t DM
        'CMIN': {'Manure': 5, 'Inter': 50, 'Grass': 50, 'Corn': 65},  # EUR/t DM
        'CMAX': {'Manure': 10, 'Inter': 80, 'Grass': 80, 'Corn': 110},  # EUR/t DM
        'YIELD': {'Manure': 0, 'Inter': 2.8, 'Grass': 12, 'Corn': 15},  # t/ha, not used
    }
    substrate_availability = {  # FM (m3 or t)
        'L1': {'Manure': 4511, 'Inter': 370, 'Grass': 246, 'Corn': 232},
        'L2': {'Manure': 1100, 'Inter': 90, 'Grass': 60, 'Corn': 57},
        'L3': {'Manure': 2089, 'Inter': 172, 'Grass': 114, 'Corn': 107},
        'L4': {'Manure': 1867, 'Inter': 154, 'Grass': 102, 'Corn': 96},
        'L5': {'Manure': 878, 'Inter': 72, 'Grass': 48, 'Corn': 45},
        'L6': {'Manure': 1100, 'Inter': 90, 'Grass': 60, 'Corn': 57},
        'L7': {'Manure': 1756, 'Inter': 144, 'Grass': 96, 'Corn': 91},
        'L8': {'Manure': 2200, 'Inter': 180, 'Grass': 120, 'Corn': 113},
    }
    full_load_hours = 7800  # h
    deprecation = 15  # y
    plants = ['K80', 'K160', 'K250', 'K500']
    plant_common_data = {
        'SYSTEM_POWER_ELECTRICITY': {'K80': 80, 'K160': 160, 'K250': 250, 'K500': 500},  # kW
        'SYSTEM_POWER_HEAT': {'K80': 105, 'K160': 230, 'K250': 300, 'K500': 560},  # kW
        'ELECTRICITY_REQUIREMENT_FACTOR': {'K80': 0.06, 'K160': 0.06, 'K250': 0.06, 'K500': 0.06},  # req / power
        'SELL_PRICE_ELECTRICITY': {'K80': 205, 'K160': 205, 'K250': 205, 'K500': 185},  # EUR/MWh
        'INVESTMENT_COST_CHP': {'K80': 110000, 'K160': 200000, 'K250': 250000, 'K500': 430000},  # EUR
        'OPERATION_COST_FERMENTER': {'K80': 18888, 'K160': 23300, 'K250': 29213, 'K500': 46706},  # EUR/y
        'OPERATION_COST_CHP': {'K80': 27138, 'K160': 33098, 'K250': 37778, 'K500': 51346},  # EUR/y
    }
    plant_specific_coefficients = {
        'HEAT_DEMAND': {
            'Manure': {'K80': 0.041157306, 'K160': 0.041157306, 'K250': 0.041157306, 'K500': 0.041157306},
            'Inter': {'K80': 0.035344135, 'K160': 0.035344135, 'K250': 0.035344135, 'K500': 0.035344135},
            'Grass': {'K80': 0.015781997, 'K160': 0.015781997, 'K250': 0.015781997, 'K500': 0.015781997},
            'Corn': {'K80': 0.041589403, 'K160': 0.041589403, 'K250': 0.041589403, 'K500': 0.041589403},
        },
        'INVESTMENT_COST_FERMENTER': {
            'Manure': {'K80': 59.29353126, 'K160': 55.94825426, 'K250': 47.66652751, 'K500': 47.18984196},
            'Inter': {'K80': 187.1176501, 'K160': 152.0278047, 'K250': 122.0870749, 'K500': 103.1706932},
            'Grass': {'K80': 246.8401822, 'K160': 196.9822356, 'K250': 153.4781289, 'K500': 88.58720996},
            'Corn': {'K80': 267.4696723, 'K160': 210.540761, 'K250': 178.7759041, 'K500': 134.8730663},
        },
    }
    substrate_min_ratios = {'Manure': 0.3}
    transformer_cost = 35000  # EUR
    electricity_purchase_price = 150  # EUR/MWh
    fermenter_heating_cost = 50  # EUR/MWh
    silo_plate_investment_cost = 150000  # EUR
    silo_plate_operating_cost = 5850  # EUR/y
    transporting_costs_fixed = {'Manure': 1, 'Inter': 2, 'Grass': 2, 'Corn': 2}  # EUR/FM
    transporting_costs_proportional = {'Manure': 0.25, 'Inter': 0.49, 'Grass': 0.49, 'Corn': 0.49}  # EUR/FM/km
    biogas_pipe_proportional_investment_cost = 22100  # EUR/km
    biogas_pipe_fixed_investment_cost = {'Pipe1': 20000, 'Pipe2': 20000, 'Pipe3to1': 60000}  # EUR
    heat_pipe_proportional_investment_cost = 300000  # EUR/km
    heat_pipe_transfer_loss = 105  # MWh/km
    heat_pipe_electricity_need = 0.005  # ratio (per heat fed in)
    heat_sell_price = 22.5  # EUR/MWh
    chp_needs_methane = 2000  # m3/kW

    # substrate transportation
    substrate_cost_use_cmin = 0
    substrate_cost_use_cmax = 1
    substrate_used_cost = {}
    for t in substrates:
        substrate_used_cost[t] = (substrate_cost_use_cmin * substrate_data['CMIN'][t] +
                                  substrate_cost_use_cmax * substrate_data['CMAX'][t])
    substrate_available_fm = {}
    for s in suppliers:
        substrate_available_fm[s] = {}
        for t in substrates:
            value_fm = substrate_availability[s][t]
            if not substrate_data['AS_FM'][t]:
                value_fm /= substrate_data['DM'][t]
            substrate_available_fm[s][t] = value_fm
    for s in suppliers:
        for t in substrates:
            substrate_fresh_name = 'substrate_available_FM_{}_{}'.format(s, t)
            substrate_fresh_price = substrate_data['DM'][t] * substrate_used_cost[t]
            problem.add_raw(substrate_fresh_name,  # available substrate, t or m3, fresh matter
                            price=substrate_fresh_price,
                            flow_rate_upper_bound=substrate_available_fm[s][t])
    for loc in locations:
        for t in substrates:
            node_name = 'substrate_feed_{}_{}'.format(loc, t)
            problem.add_inter(node_name)  # delivered substrate, t or m3, fresh matter
    for s in suppliers:
        for loc in locations:
            for t in substrates:
                transport_unit_name = 'op_transport_substrates_{}_{}_{}'.format(s, loc, t)
                transport_cost = (transporting_costs_fixed[t] +
                                  transporting_costs_proportional[t] * distance[s][loc])
                problem.add_op_unit(transport_unit_name,
                                    proportional_cost=transport_cost)
                input_name = 'substrate_available_FM_{}_{}'.format(s, t)
                problem.add_in(input_name, transport_unit_name)
                output_name = 'substrate_feed_{}_{}'.format(loc, t)
                problem.add_out(transport_unit_name, output_name)
    # fermenter requirements
    problem.add_raw('heat_purchase',  # input heating possibly needed by fermenters, MWh
                    price=fermenter_heating_cost)
    for loc in locations:
        heat_purchase_op_unit_name = 'purchase_heat_{}'.format(loc)
        problem.add_op_unit(heat_purchase_op_unit_name)
        input_name = 'heat_purchase'
        problem.add_in(input_name, heat_purchase_op_unit_name)
        output_name = 'heat_balance_{}'.format(loc)
        problem.add_out(heat_purchase_op_unit_name, output_name)
    for loc in chp_locations:
        biogas_node_name = 'biogas_{}'.format(loc)
        problem.add_inter(biogas_node_name)  # biogas balance at location, m3
        heat_balance_node_name = 'heat_balance_{}'.format(loc)
        problem.add_inter(heat_balance_node_name)  # heat balance at location, MWh
    # silo plates
    for loc in locations:
        silo_plate_op_unit_name = 'silo_plate_{}'.format(loc)
        silo_plate_cost = silo_plate_investment_cost / deprecation + silo_plate_operating_cost
        problem.add_op_unit(silo_plate_op_unit_name,
                            fix_cost=silo_plate_cost)
        silo_plate_capacity_name = 'silo_plate_availability_material_{}'.format(loc)
        problem.add_inter(silo_plate_capacity_name)
        problem.add_out(silo_plate_op_unit_name, silo_plate_capacity_name)
    # calculating the most expensive "slack substrate" for investment into unfilled capacity
    slack_substrates = {}
    for k in plants:
        def condition(t_slack):
            return (plant_specific_coefficients['INVESTMENT_COST_FERMENTER'][t_slack][k]
                    / (substrate_data['DM'][t_slack] * substrate_data['CH4'][t_slack]))
        slack_substrates[k] = max(substrates, key=condition)
    # fermenters
    for loc in locations:
        for k in plants:
            for i in range(max_identical_fermenters):
                fermenter_investment_unit_name = 'invest_fermenter_{}_{}_No{}'.format(loc, k, i)
                problem.add_op_unit(fermenter_investment_unit_name,
                                    fix_cost=plant_common_data['OPERATION_COST_FERMENTER'][k],
                                    capacity_lower_bound=full_load_hours,
                                    capacity_upper_bound=full_load_hours)
                capacity_output_name = 'capacity_out_{}_{}_No{}'.format(loc, k, i)
                problem.add_inter(capacity_output_name)
                capacity_input_name = 'capacity_in_{}_{}_No{}'.format(loc, k, i)
                problem.add_inter(capacity_input_name)
                problem.add_in(capacity_output_name, fermenter_investment_unit_name)
                problem.add_out(fermenter_investment_unit_name, capacity_input_name)
                silo_plate_capacity_name = 'silo_plate_availability_material_{}'.format(loc)
                problem.add_in(silo_plate_capacity_name, fermenter_investment_unit_name)
                for t in substrates:
                    fermenter_process_name = 'process_{}_{}_No{}_{}'.format(loc, k, i, t)
                    investment_cost = plant_specific_coefficients['INVESTMENT_COST_FERMENTER'][t][k] / deprecation
                    problem.add_op_unit(fermenter_process_name,
                                        proportional_cost=investment_cost)
                    biomass_input_name = 'substrate_feed_{}_{}'.format(loc, t)
                    problem.add_in(biomass_input_name, fermenter_process_name,
                                   flow=1)
                    biogas_output_name = 'biogas_{}'.format(loc)
                    biogas_per_feed = substrate_data['DM'][t] * substrate_data['CH4'][t]
                    problem.add_out(fermenter_process_name, biogas_output_name,
                                    flow=biogas_per_feed)
                    heat_input_name = 'heat_balance_{}'.format(loc)
                    heat_requirement_per_feed = plant_specific_coefficients['HEAT_DEMAND'][t][k]
                    problem.add_in(heat_input_name, fermenter_process_name,
                                   flow=heat_requirement_per_feed)
                    capacity_need_per_biogas = (biogas_per_feed * full_load_hours / chp_needs_methane
                                                / plant_common_data['SYSTEM_POWER_ELECTRICITY'][k])
                    problem.add_in(capacity_input_name, fermenter_process_name,
                                   flow=capacity_need_per_biogas)
                    problem.add_out(fermenter_process_name, capacity_output_name,
                                    flow=capacity_need_per_biogas)
                for t in [slack_substrates[k]]:
                    fermenter_process_name = 'process_{}_{}_No{}_slack'.format(loc, k, i)
                    investment_cost = plant_specific_coefficients['INVESTMENT_COST_FERMENTER'][t][k]
                    problem.add_op_unit(fermenter_process_name,
                                        proportional_cost=investment_cost)
                    biogas_per_feed = substrate_data['DM'][t] * substrate_data['CH4'][t]
                    capacity_need_per_biogas = (biogas_per_feed * full_load_hours / chp_needs_methane
                                                / plant_common_data['SYSTEM_POWER_ELECTRICITY'][k])
                    problem.add_in(capacity_input_name, fermenter_process_name,
                                   flow=capacity_need_per_biogas)
                    problem.add_out(fermenter_process_name, capacity_output_name,
                                    flow=capacity_need_per_biogas)
                for t in substrate_min_ratios:
                    ratio = substrate_min_ratios[t]
                    constraint_material_name = 'min_ratio_{}_{}_No{}_{}'.format(loc, k, i, t)
                    problem.add_inter(constraint_material_name)
                    for t2 in substrates:
                        fermenter_process_name = 'process_{}_{}_No{}_{}'.format(loc, k, i, t2)
                        if t2 == t:
                            problem.add_out(fermenter_process_name, constraint_material_name,
                                            flow=(1 - ratio))
                        else:
                            problem.add_in(constraint_material_name, fermenter_process_name,
                                           flow=ratio)
    # biogas pipes
    for p in pipe_sections:
        biogas_pipe_op_unit_name = 'biogas_pipe_{}'.format(p)
        biogas_pipe_investment_total = (biogas_pipe_fixed_investment_cost[p] +
                                        biogas_pipe_proportional_investment_cost * pipe_section_length[p])
        biogas_pipe_cost = biogas_pipe_investment_total / deprecation
        problem.add_op_unit(biogas_pipe_op_unit_name,
                            fix_cost=biogas_pipe_cost)
        biogas_pipe_capacity_name = 'biogas_pipe_availability_material_{}'.format(p)
        problem.add_inter(biogas_pipe_capacity_name)
        problem.add_out(biogas_pipe_op_unit_name, biogas_pipe_capacity_name)
    for loc in locations:
        biogas_transfer_op_unit_name = 'transfer_biogas_{}'.format(loc)
        problem.add_op_unit(biogas_transfer_op_unit_name)
        for p in pipe_requirements[loc]:
            biogas_pipe_capacity_name = 'biogas_pipe_availability_material_{}'.format(p)
            problem.add_in(biogas_pipe_capacity_name, biogas_transfer_op_unit_name)
        biogas_start_name = 'biogas_{}'.format(loc)
        problem.add_in(biogas_start_name, biogas_transfer_op_unit_name)
        biogas_center_name = 'biogas_{}'.format('CENTER')
        problem.add_out(biogas_transfer_op_unit_name, biogas_center_name)
    # heat pipes
    for loc in locations:
        heat_transfer_op_unit_name = 'transfer_heat_{}'.format(loc)
        heat_transfer_cost = heat_pipe_electricity_need * electricity_purchase_price
        problem.add_op_unit(heat_transfer_op_unit_name,
                            proportional_cost=heat_transfer_cost)
        source_heat_material_name = 'heat_balance_{}'.format(loc)
        target_heat_material_name = 'heat_sent_{}'.format(loc)
        final_heat_material_name = 'heat_balance_{}'.format('CENTER')
        problem.add_inter(target_heat_material_name)  # MWh
        problem.add_in(source_heat_material_name, heat_transfer_op_unit_name)
        problem.add_out(heat_transfer_op_unit_name, target_heat_material_name)
        heat_transfer_finishing_name = 'save_heat_{}'.format(loc)
        problem.add_op_unit(heat_transfer_finishing_name)
        problem.add_in(target_heat_material_name, heat_transfer_finishing_name)
        problem.add_out(heat_transfer_finishing_name, final_heat_material_name)
        for p in pipe_requirements[loc]:
            heat_pipe_capacity_name = 'heat_pipe_availability_material_{}_{}'.format(loc, p)
            problem.add_inter(heat_pipe_capacity_name)
            problem.add_in(heat_pipe_capacity_name, heat_transfer_op_unit_name,
                           flow=0.001)
    for p in pipe_sections:
        heat_pipe_capacity_op_unit_name = 'heat_pipe_provide_availability_{}'.format(p)
        heat_loss_value = pipe_section_length[p] * heat_pipe_transfer_loss
        heat_pipe_cost = heat_pipe_proportional_investment_cost * pipe_section_length[p] / deprecation
        problem.add_op_unit(heat_pipe_capacity_op_unit_name,
                            fix_cost=heat_pipe_cost,
                            capacity_lower_bound=heat_loss_value)
        collected_heat_loss_material_name = 'collected_heat_loss_{}'.format(p)
        problem.add_inter(collected_heat_loss_material_name)
        problem.add_in(collected_heat_loss_material_name, heat_pipe_capacity_op_unit_name)
        for loc in locations:
            if p in pipe_requirements[loc]:
                heat_pipe_capacity_name = 'heat_pipe_availability_material_{}_{}'.format(loc, p)
                problem.add_out(heat_pipe_capacity_op_unit_name, heat_pipe_capacity_name,
                                flow=1e6)
                collect_heat_loss_op_unit_name = 'collect_heat_loss_op_unit_{}_{}'.format(loc, p)
                problem.add_op_unit(collect_heat_loss_op_unit_name)
                target_heat_material_name = 'heat_sent_{}'.format(loc)
                problem.add_in(target_heat_material_name, collect_heat_loss_op_unit_name)
                problem.add_out(collect_heat_loss_op_unit_name, collected_heat_loss_material_name)
    # transformer unit
    transformer_op_unit_name = 'transformer_op_unit'
    transformer_op_unit_fix_cost = transformer_cost / deprecation
    problem.add_op_unit(transformer_op_unit_name,
                        fix_cost=transformer_op_unit_fix_cost)
    transformer_capacity_name = 'transformer_availability_material'
    problem.add_inter(transformer_capacity_name)
    problem.add_out(transformer_op_unit_name, transformer_capacity_name)
    # chp units
    for loc in chp_locations:
        for k in plants:
            for j in range(max_identical_chp):
                chp_fix_cost = (plant_common_data['INVESTMENT_COST_CHP'][k] / deprecation +
                                plant_common_data['OPERATION_COST_CHP'][k])
                per_hour_biogas_input = (chp_needs_methane * plant_common_data['SYSTEM_POWER_ELECTRICITY'][k]
                                         / full_load_hours)
                per_hour_electricity_output = plant_common_data['SYSTEM_POWER_ELECTRICITY'][k] / 1000
                per_hour_heat_output = plant_common_data['SYSTEM_POWER_HEAT'][k] / 1000
                per_hour_cost = (per_hour_electricity_output * electricity_purchase_price
                                 * plant_common_data['ELECTRICITY_REQUIREMENT_FACTOR'][k])
                chp_unit_name = 'chp_{}_{}_No{}'.format(loc, k, j)
                problem.add_op_unit(chp_unit_name,
                                    fix_cost=chp_fix_cost,
                                    proportional_cost=per_hour_cost,
                                    capacity_upper_bound=full_load_hours)
                biogas_input_name = 'biogas_{}'.format(loc)
                electricity_output_name = 'electricity_{}'.format(k)
                heat_output_name = 'heat_balance_{}'.format(loc)
                problem.add_in(transformer_capacity_name, chp_unit_name)
                problem.add_in(biogas_input_name, chp_unit_name,
                               flow=per_hour_biogas_input)
                problem.add_out(chp_unit_name, electricity_output_name,
                                flow=per_hour_electricity_output)
                problem.add_out(chp_unit_name, heat_output_name,
                                flow=per_hour_heat_output)
    # selling energy
    revenue_node_name = 'revenue'
    problem.add_product(revenue_node_name,  # EUR
                        price=1)
    for k in plants:
        electricity_material_name = 'electricity_{}'.format(k)
        problem.add_inter(electricity_material_name)  # MWh
        sell_electricity_op_unit_name = 'sell_electricity_{}'.format(k)
        problem.add_op_unit(sell_electricity_op_unit_name)
        problem.add_in(electricity_material_name, sell_electricity_op_unit_name)
        problem.add_out(sell_electricity_op_unit_name, revenue_node_name,
                        flow=plant_common_data['SELL_PRICE_ELECTRICITY'][k])
    sell_heat_op_unit_name = 'sell_heat'
    problem.add_op_unit(sell_heat_op_unit_name)
    final_heat_material_name = 'heat_balance_{}'.format('CENTER')
    problem.add_in(final_heat_material_name, sell_heat_op_unit_name)
    problem.add_out(sell_heat_op_unit_name, revenue_node_name,
                    flow=heat_sell_price)

    return problem
