class Problem:
    def __init__(self):
        self.materials = {}
        self.op_units = {}
        self.inflows = {}
        self.outflows = {}
        self.producing = {}
        self.consuming = {}
        self.supported_material_attributes = [
            'price', 'flow_rate_upper_bound', 'flow_rate_lower_bound'
        ]
        self.supported_op_unit_attributes = [
            'capacity_lower_bound', 'capacity_upper_bound', 'fix_cost', 'proportional_cost'
        ]
        self.problem_settings = {
            'file_type': 'PNS_problem_v1',
            'file_name': 'FILE_NAME',
            'mass_unit': 'kg',
            'time_unit': 'y',
            'money_unit': 'HUF',
            'material_type': 'intermediate',
            'material_flow_rate_lower_bound': 0,
            'material_flow_rate_upper_bound': 1000000000,
            'material_price': 0,
            'operating_unit_capacity_lower_bound': 0,
            'operating_unit_capacity_upper_bound': 1000000000,
            'operating_unit_fix_cost': 0,
            'operating_unit_proportional_cost': 0
        }

    def set(self, name, value):
        assert name in self.problem_settings,\
            f'unknown problem setting "{name}" to be set to value "{value}"'
        self.problem_settings[name] = value

    def get(self, name):
        assert name in self.problem_settings,\
            f'unknown problem setting "{name}" queried'
        return self.problem_settings[name]

    def add_material(self, material_name, material_type, **material_attributes):
        assert material_name,\
            'material name cannot be empty'
        assert material_name not in self.materials,\
            f'material "{material_name}" already exists'
        assert material_type in ['raw_material', 'intermediate', 'product'],\
            f'bad type "{material_type}" for material "{material_name}"'
        for attribute in material_attributes:
            assert attribute in self.supported_material_attributes,\
                f'unsupported material attribute "{attribute}"'
        material = {
            'name': material_name,
            'type': material_type,
        }
        for supported_attribute in self.supported_material_attributes:
            material[supported_attribute] = material_attributes.get(supported_attribute)
        self.materials[material_name] = material
        self.producing.setdefault(material_name, {})
        self.consuming.setdefault(material_name, {})

    def add_raw(self, raw_name, **material_attributes):
        self.add_material(raw_name, 'raw_material', **material_attributes)

    def add_inter(self, inter_name, **material_attributes):
        self.add_material(inter_name, 'intermediate', **material_attributes)

    def add_product(self, product_name, **material_attributes):
        self.add_material(product_name, 'product', **material_attributes)

    def add_op_unit(self, op_unit_name, **op_unit_attributes):
        assert op_unit_name,\
            'operating unit name cannot be empty'
        assert op_unit_name not in self.op_units,\
            f'operating unit "{op_unit_name}" already exists'
        for attribute in op_unit_attributes:
            assert attribute in self.supported_op_unit_attributes,\
                f'unsupported operating unit attribute "{attribute}"'
        op_unit = {
            'name': op_unit_name,
        }
        for supported_attribute in self.supported_op_unit_attributes:
            op_unit[supported_attribute] = op_unit_attributes.get(supported_attribute)
        self.op_units[op_unit_name] = op_unit
        self.inflows.setdefault(op_unit_name, {})
        self.outflows.setdefault(op_unit_name, {})

    def add_in(self, material_name, op_unit_name, flow=1):
        self.inflows.setdefault(op_unit_name, {})
        self.consuming.setdefault(material_name, {})
        assert material_name not in self.inflows[op_unit_name],\
            f'inflow already defined from "{material_name}" to "{op_unit_name}"'
        self.inflows[op_unit_name][material_name] = flow
        self.consuming[material_name][op_unit_name] = flow

    def add_out(self, op_unit_name, material_name, flow=1):
        self.outflows.setdefault(op_unit_name, {})
        self.producing.setdefault(material_name, {})
        assert material_name not in self.outflows[op_unit_name],\
            f'outflow already defined from "{op_unit_name}" to "{material_name}"'
        self.outflows[op_unit_name][material_name] = flow
        self.producing[material_name][op_unit_name] = flow

    def get_materials_text(self):
        result = ''
        for material_name, material_properties in sorted(self.materials.items()):
            result += '{}: {}'.format(material_name, material_properties['type'])
            for property_name in self.supported_material_attributes:
                property_value = material_properties.get(property_name)
                if property_value is not None:
                    result += ', {}={}'.format(property_name, property_value)
            result += '\n'
        return result

    def get_op_units_text(self):
        result = ''
        for op_unit_name, op_unit_properties in sorted(self.op_units.items()):
            result += '{}'.format(op_unit_name)
            delimiter = ': '
            for property_name in self.supported_op_unit_attributes:
                property_value = op_unit_properties.get(property_name)
                if property_value is not None:
                    result += '{}{}={}'.format(delimiter, property_name, property_value)
                    delimiter = ', '
            result += '\n'
        return result

    def get_flow_rates_text(self):
        result = ''
        for op_unit_name in sorted(self.op_units.keys()):
            inputs = self.inflows.get(op_unit_name)
            outputs = self.outflows.get(op_unit_name)
            if not (inputs or outputs):
                continue
            if not inputs:
                inputs = {}
            if not outputs:
                outputs = {}
            result += '{}:'.format(op_unit_name)
            delimiter = ''
            for input_material_name, input_flow_rate in inputs.items():
                if input_flow_rate == 1:
                    result += '{} {}'.format(delimiter, input_material_name)
                else:
                    result += '{} {} {}'.format(delimiter, input_flow_rate, input_material_name)
                delimiter = ' +'
            result += " =>"
            delimiter = ''
            for output_material_name, output_flow_rate in outputs.items():
                if output_flow_rate == 1:
                    result += '{} {}'.format(delimiter, output_material_name)
                else:
                    result += '{} {} {}'.format(delimiter, output_flow_rate, output_material_name)
                delimiter = ' +'
            result += '\n'
        return result

    def get_init_text(self):
        def setting_text(name, delimiter='='):
            return f'{name}{delimiter}{self.problem_settings[name]}\n'
        init_text = (f'{setting_text("file_type")}'
                     f'{setting_text("file_name")}'
                     '\n'
                     'measurement_units:\n'
                     f'{setting_text("mass_unit")}'
                     f'{setting_text("time_unit")}'
                     f'{setting_text("money_unit")}'
                     '\n'
                     'defaults:\n'
                     f'{setting_text("material_type")}'
                     f'{setting_text("material_flow_rate_lower_bound")}'
                     f'{setting_text("material_flow_rate_upper_bound")}'
                     f'{setting_text("material_price")}'
                     f'{setting_text("operating_unit_capacity_lower_bound")}'
                     f'{setting_text("operating_unit_capacity_upper_bound")}'
                     f'{setting_text("operating_unit_fix_cost")}'
                     f'{setting_text("operating_unit_proportional_cost")}'
                     )
        return init_text

    def get_problem_text(self):
        problem_text = (f'{self.get_init_text()}\n'
                        'materials:\n'
                        f'{self.get_materials_text()}'
                        '\n'
                        'operating_units:\n'
                        f'{self.get_op_units_text()}'
                        '\n'
                        'material_to_operating_unit_flow_rates:\n'
                        f'{self.get_flow_rates_text()}'
                        '\n'
                        )
        return problem_text

    def check_structure(self):
        for material_name in self.materials:
            assert material_name not in self.op_units,\
                f'material name "{material_name}" is also an operating unit name'
        for op_unit_name in self.inflows:
            assert op_unit_name in self.op_units,\
                f'inflow operating unit "{op_unit_name}" not found'
            for material_name in self.inflows[op_unit_name]:
                assert material_name in self.materials,\
                    f'inflow material "{material_name}" not found, to "{op_unit_name}"'
        for op_unit_name in self.outflows:
            assert op_unit_name in self.op_units,\
                f'outflow operating unit "{op_unit_name}" not found'
            for material_name in self.outflows[op_unit_name]:
                assert material_name in self.materials,\
                    f'outflow material "{material_name} not found, from "{op_unit_name}"'
        # redundant from this point due to implementation
        for material_name in self.producing:
            assert material_name in self.materials,\
                f'produced material "{material_name}" not found'
            for op_unit_name in self.producing[material_name]:
                assert op_unit_name in self.op_units,\
                    f'operating unit "{op_unit_name} producing "{material_name}" not found'
        for material_name in self.consuming:
            assert material_name in self.materials,\
                f'consumed material "{material_name}" not found'
            for op_unit_name in self.consuming[material_name]:
                assert op_unit_name in self.op_units,\
                    f'operating unit "{op_unit_name} consuming "{material_name}" not found'
        for material_name in self.materials:
            assert material_name in self.producing,\
                f'material "{material_name}" not found in "producing" dict'
            assert material_name in self.consuming,\
                f'material "{material_name}" not found in "producing" dict'
        for op_unit_name in self.op_units:
            assert op_unit_name in self.inflows,\
                f'operating unit "{op_unit_name}" not found in "inflows" dict'
            assert op_unit_name in self.outflows,\
                f'operating unit "{op_unit_name}" not found in "outflows" dict'

    def get_stats(self):
        material_count = len(self.materials)
        op_unit_count = len(self.op_units)
        inflow_count = 0
        for inflow_material_and_flow in self.inflows.values():
            inflow_count += len(inflow_material_and_flow)
        outflow_count = 0
        for outflow_material_and_flow in self.outflows.values():
            outflow_count += len(outflow_material_and_flow)
        non_raw_materials_not_produced = []
        raw_materials_produced = []
        non_product_materials_not_consumed = []
        product_materials_consumed = []
        for material in self.materials:
            is_produced = bool(self.producing[material])
            is_consumed = bool(self.consuming[material])
            material_type = self.materials[material]['type']
            if material_type == 'raw_material':
                if is_produced:
                    raw_materials_produced.append(material)
            else:
                if not is_produced:
                    non_raw_materials_not_produced.append(material)
            if material_type == 'product':
                if is_consumed:
                    product_materials_consumed.append(material)
            else:
                if not is_consumed:
                    non_product_materials_not_consumed.append(material)
        contribution_depth_material = {}
        contribution_depth_op_unit = {}
        to_be_processed = [material for material in self.materials if self.materials[material]['type'] == 'product']
        for material in to_be_processed:
            contribution_depth_material[material] = 0
        depth = 1
        while to_be_processed:
            process_next = []
            if depth % 2:
                for material in to_be_processed:
                    for op_unit in self.producing[material]:
                        if op_unit not in contribution_depth_op_unit:
                            contribution_depth_op_unit[op_unit] = depth
                            process_next.append(op_unit)
            else:
                for op_unit in to_be_processed:
                    for material in self.inflows[op_unit]:
                        if material not in contribution_depth_material:
                            contribution_depth_material[material] = depth
                            process_next.append(material)
            to_be_processed = process_next
            depth += 1
        materials_without_contribution = []
        op_units_without_contribution = []
        for material in self.materials:
            if material not in contribution_depth_material:
                materials_without_contribution.append(material)
        for op_unit in self.op_units:
            if op_unit not in contribution_depth_op_unit:
                op_units_without_contribution.append(op_unit)
        component_id_material = {}
        component_id_op_unit = {}
        component_id = 0
        for material in self.materials:
            if material not in component_id_material:
                component_id += 1
                component_id_material[material] = component_id
                to_be_processed = [material]
                turn_on_materials = True
                while to_be_processed:
                    process_next = []
                    if turn_on_materials:
                        for node_material in to_be_processed:
                            adj_op_unit_list = ([op_unit for op_unit in self.consuming[node_material]] +
                                                [op_unit for op_unit in self.producing[node_material]])
                            for adj_op_unit in adj_op_unit_list:
                                if adj_op_unit not in component_id_op_unit:
                                    component_id_op_unit[adj_op_unit] = component_id
                                    process_next.append(adj_op_unit)
                    else:
                        for node_op_unit in to_be_processed:
                            adj_material_list = ([material for material in self.inflows[node_op_unit]] +
                                                 [material for material in self.outflows[node_op_unit]])
                            for adj_material in adj_material_list:
                                if adj_material not in component_id_material:
                                    component_id_material[adj_material] = component_id
                                    process_next.append(adj_material)
                    to_be_processed = process_next
                    turn_on_materials = not turn_on_materials
        for op_unit in self.op_units:
            if not (self.inflows[op_unit] or self.outflows[op_unit]):
                component_id += 1
                component_id_op_unit[op_unit] = component_id
        component_count = component_id
        stats = {
            'stats': {
                'material_count': material_count,
                'op_unit_count': op_unit_count,
                'inflow_count': inflow_count,
                'outflow_count': outflow_count,
            },
            'issues': {
                'component_count': component_count,
                'non_raw_materials_not_produced': non_raw_materials_not_produced,
                'raw_materials_produced': raw_materials_produced,
                'non_product_materials_not_consumed': non_product_materials_not_consumed,
                'product_materials_consumed': product_materials_consumed,
                'materials_without_contribution': materials_without_contribution,
                'op_units_without_contribution': op_units_without_contribution,
            },
            'contribution_depth': {
                'contribution_depth_material': contribution_depth_material,
                'contribution_depth_op_unit': contribution_depth_op_unit,
            },
            'component_id': {
                'component_id_material': component_id_material,
                'component_id_op_unit': component_id_op_unit,
            },
        }
        return stats

    def print_stats(self, *args):
        stats = self.get_stats()
        for stat_type in args:
            print(f'Stats "{stat_type}":')
            current_stats = stats[stat_type]
            for key, value in current_stats.items():
                print(f'{key} -> {value}')
