# Model for Bad Zell Case Study
# Original reproduced model + data merged in one file

### Sets and Parameters

set Suppliers; # group of individual suppliers

set Individual_Suppliers;

param Individual_Supplier_Group {si in Individual_Suppliers}, symbolic, in Suppliers;

set Locations;

param Distance {s in Suppliers, l in Locations}, >=0; # km

set Pipe_Sections;

param Pipe_Section_Length {p in Pipe_Sections}, >=0; # km

set Pipe_Requirements, dimen 2, within (Locations cross Pipe_Sections);

set Substrates;

set Substrates_FIELDS := {
	'DM', # dry matter content, as ratio
	'CH4', # methane (CH4) content, m3/unit DM
	'CMIN', # purchase cost, minimum EUR/t DM
	'CMAX', # purchase cost, maximum EUR/t DM
	'YIELD' # DM t/ha
};

param Substrates_data {field in Substrates_FIELDS, t in Substrates};

param Substrates_Availability {s in Suppliers, t in Substrates};

param Substrate_DM_for_Availability_Factor {t in Substrates}; # availability * factor = DM t

set Fermenter_Mixtures;

param Fermenter_Mixtures_Compositions {m in Fermenter_Mixtures, t in Substrates}, >=0, <=100; # percent
check {m in Fermenter_Mixtures} (sum {t in Substrates} Fermenter_Mixtures_Compositions[m,t] = 100);

param Full_Load_Hours; # h

param Deprecation; # y

set Plants;

set Plants_Common_FIELDS := {
	'SYSTEM_POWER_ELECTRICITY', # kW, times full load hours gives generated amount
	'SYSTEM_POWER_HEAT', # kW, times full load hours gives generated amount
	'ELECTRICITY_REQUIREMENT_FACTOR', # ratio, based on generated amount
	'SELL_PRICE_ELECTRICITY', # EUR/MWh, based on feed-in tariffs
	'INVESTMENT_COST_CHP', # EUR
	'OPERATION_COST_FERMENTER', # EUR/y
	'OPERATION_COST_CHP' # EUR/y
};

param Plants_Common_data {f in Plants_Common_FIELDS, k in Plants};

set Plants_Specific_FIELDS := {
	'HEAT_DEMAND', # MWh
	'INVESTMENT_COST_FERMENTER' # EUR
};

param Plants_Specific_data {f in Plants_Specific_FIELDS, k in Plants, m in Fermenter_Mixtures};

param Substrates_Cost_Use_CMIN, default 0.0;
param Substrates_Cost_Use_CMAX, default 1.0;

param Transformer_Cost; # EUR

param Electricity_Purchase_Price; # EUR/MWh

param Fermenter_Heating_Cost; # EUR/MWh

param Silo_Plate_Investment_Cost; # EUR

param Silo_Plate_Operating_Cost; # EUR/y

param Transporting_Costs_Fixed {t in Substrates}; # EUR/unit

param Transporting_Costs_Proportional {t in Substrates}; # EUR/unit/km

param Biogas_Pipe_Propotional_Investment_Cost; # EUR/km

param Biogas_Pipe_Fixed_Investment_Cost {p in Pipe_Sections}; # EUR

param Heat_Pipe_Proportional_Investment_Cost; # EUR/km

param Heat_Pipe_Transfer_Loss; # MWh/km

param Heat_Pipe_Electricity_Need; # ratio per heat fed in

param Heat_Sell_Price; # EUR/MWh

### Additional constants

param CHP_Needs_Methane, default 2000; # m3/kW_el (ratio of methane needed for a CHP to maintain an electric output for a year)

param Max_Identical_Equipment, >=0, default 3; # count

param Max_Methane_Transport, default 2000000; # m3

param Max_Heat_Transport, default 20000; # MWh

### Variables

var v_use_centralized_chp {k in Plants}, integer, >=0, <=Max_Identical_Equipment;
var v_use_decentralized_chp {k in Plants, l in Locations}, integer, >=0, <=Max_Identical_Equipment;
var v_use_fermenter {k in Plants, m in Fermenter_Mixtures, l in Locations}, integer, >=0, <=Max_Identical_Equipment;

var v_use_silo_plate {l in Locations}, binary;
var v_use_transformer, binary;
var v_is_biogas_transport {l in Locations}, binary;
var v_is_heat_transport {l in Locations}, binary;
var v_use_biogas_pipe {p in Pipe_Sections}, binary;
var v_use_heat_pipe {p in Pipe_Sections}, binary;

var v_transport_substrates {s in Suppliers, l in Locations, t in Substrates}, >=0; # unit (m3 or t) fresh
var v_substrate_collected {l in Locations, t in Substrates}, >=0; # unit (m3 or t) fresh
var v_substrate_feed {k in Plants, m in Fermenter_Mixtures, l in Locations, t in Substrates}, >=0; # unit (m3 or t) fresh
var v_methane_produced {k in Plants, m in Fermenter_Mixtures, l in Locations}, >=0; # m3
var v_fermenter_hours {k in Plants, m in Fermenter_Mixtures, l in Locations}, >=0; # h
var v_transport_methane {l in Locations}, >=0; # m3 (to centre)
var v_chp_decentralized_hours {k in Plants, l in Locations}, >=0; # h
var v_chp_centralized_hours {k in Plants}, >=0; # h
var v_fermenter_heating_purchased {l in Locations}, >=0; # MWh
var v_transport_heat {l in Locations}, >=0; # MWh
var v_heat_loss {p in Pipe_Sections}, >=0; # MWh
var v_heat_loss_cover {(l,p) in Pipe_Requirements}, >=0; # MWh
var v_sell_heat; # MWh
var v_sell_electricity {k in Plants}; # MWh

### Constraints: Material Balance

param Substrate_Available_FM {s in Suppliers, t in Substrates} :=
	Substrates_Availability[s,t] * Substrate_DM_for_Availability_Factor[t] / Substrates_data['DM',t];
s.t. C_Transport_Substrates_Limit {s in Suppliers, t in Substrates}:
	sum {l in Locations} v_transport_substrates[s,l,t] <= Substrate_Available_FM[s,t];

s.t. C_Substrate_Feed_Calculation {l in Locations, t in Substrates}:
	sum {s in Suppliers} v_transport_substrates[s,l,t] = sum {k in Plants, m in Fermenter_Mixtures} v_substrate_feed[k,m,l,t];

param Substrate_Methane_Coef {m in Fermenter_Mixtures, t in Substrates} :=
	(Fermenter_Mixtures_Compositions[m,t] / 100) * Substrates_data['DM',t] * Substrates_data['CH4',t];
param Substrate_Methane_Sum {m in Fermenter_Mixtures} := sum {t in Substrates} Substrate_Methane_Coef[m,t];
s.t. C_Methane_Produced_Calculation {k in Plants, m in Fermenter_Mixtures, l in Locations, t in Substrates}:
	v_methane_produced[k,m,l] * (Substrate_Methane_Coef[m,t] / Substrate_Methane_Sum[m])
	= v_substrate_feed[k,m,l,t] * Substrates_data['DM',t] * Substrates_data['CH4',t];

s.t. C_Fermenter_Hours_Calculation {k in Plants, m in Fermenter_Mixtures, l in Locations}:
	(v_fermenter_hours[k,m,l] / Full_Load_Hours) * CHP_Needs_Methane * Plants_Common_data['SYSTEM_POWER_ELECTRICITY',k] = v_methane_produced[k,m,l];

s.t. C_Methane_Usage_Balance_In_Locations {l in Locations}:
	(sum {k in Plants, m in Fermenter_Mixtures} v_methane_produced[k,m,l])
	= v_transport_methane[l]
	+ sum {k in Plants} (v_chp_decentralized_hours[k,l] / Full_Load_Hours) * CHP_Needs_Methane * Plants_Common_data['SYSTEM_POWER_ELECTRICITY',k];
	# wasting methane allowed? NO

s.t. C_Methane_Usage_Balance_In_Center:
	sum {l in Locations} v_transport_methane[l] =
	sum {k in Plants} (v_chp_centralized_hours[k] / Full_Load_Hours) * CHP_Needs_Methane * Plants_Common_data['SYSTEM_POWER_ELECTRICITY',k];

s.t. C_Heat_Balance_In_Locations {l in Locations}:
	sum {k in Plants} v_chp_decentralized_hours[k,l] * (Plants_Common_data['SYSTEM_POWER_HEAT',k] / 1000)
	+ v_fermenter_heating_purchased[l]
	= sum {k in Plants, m in Fermenter_Mixtures} (v_fermenter_hours[k,m,l] / Full_Load_Hours) * Plants_Specific_data['HEAT_DEMAND',k,m]
	+ v_transport_heat[l];
	# wasting heat allowed? NO

s.t. C_Heat_Loss_Calculation_Location_Side {l in Locations}:
	v_transport_heat[l] >= sum {(l,p) in Pipe_Requirements} v_heat_loss_cover[l,p];

s.t. C_Heat_Loss_Calculation_Pipe_Side {p in Pipe_Sections}:
	v_heat_loss[p] = sum {(l,p) in Pipe_Requirements} v_heat_loss_cover[l,p];

s.t. C_Sell_Heat_Calculation:
	v_sell_heat =
	sum {l in Locations} v_transport_heat[l]
	- sum {(l,p) in Pipe_Requirements} v_heat_loss_cover[l,p]
	+ sum {k in Plants} v_chp_centralized_hours[k] * (Plants_Common_data['SYSTEM_POWER_HEAT',k] / 1000);

s.t. C_Sell_Electricity_Calculation {k in Plants}:
	v_sell_electricity[k] =
	(Plants_Common_data['SYSTEM_POWER_ELECTRICITY',k] / 1000) *
	(v_chp_centralized_hours[k] + sum {l in Locations} v_chp_decentralized_hours[k,l]);

### Constraints: Logical

s.t. C_Usage_of_Centralized_CHP {k in Plants}:
	v_chp_centralized_hours[k] <= v_use_centralized_chp[k] * Full_Load_Hours;

s.t. C_Usage_of_Decentralized_CHP {k in Plants, l in Locations}:
	v_chp_decentralized_hours[k,l] <= v_use_decentralized_chp[k,l] * Full_Load_Hours;

s.t. C_Usage_of_Fermenters {k in Plants, m in Fermenter_Mixtures, l in Locations}:
	v_fermenter_hours[k,m,l] <= v_use_fermenter[k,m,l] * Full_Load_Hours;

s.t. C_Usage_of_Silo_Plate {k in Plants, m in Fermenter_Mixtures, l in Locations}:
	v_use_silo_plate[l] * Max_Identical_Equipment >= v_use_fermenter[k,m,l];

s.t. C_Usage_of_Transformer_1 {k in Plants}:
	v_use_transformer * Max_Identical_Equipment >= v_use_centralized_chp[k];

s.t. C_Usage_of_Transformer_2 {k in Plants, l in Locations}:
	v_use_transformer * Max_Identical_Equipment >= v_use_decentralized_chp[k,l];

s.t. C_Usage_of_Biogas_Transport {l in Locations}:
	v_is_biogas_transport[l] * Max_Methane_Transport >= v_transport_methane[l];

s.t. C_Usage_of_Heat_Transport {l in Locations}:
	v_is_heat_transport[l] * Max_Heat_Transport >= v_transport_heat[l];

s.t. C_Usage_of_Biogas_Pipe {(l,p) in Pipe_Requirements}:
	v_use_biogas_pipe[p] >= v_is_biogas_transport[l];

s.t. C_Usage_of_Heat_Pipe {(l,p) in Pipe_Requirements}:
	v_use_heat_pipe[p] >= v_is_heat_transport[l];

s.t. C_Is_Heat_Loss {p in Pipe_Sections}:
	v_heat_loss[p] >= v_use_heat_pipe[p] * Pipe_Section_Length[p] * Heat_Pipe_Transfer_Loss;

### Constraints: Incomes

var vo_income_heat; # EUR/y
var vo_income_electricity; # EUR/y
var vo_yearly_income; # EUR/y

s.t. C_Income_From_Heat_Calculation: vo_income_heat =
	v_sell_heat * Heat_Sell_Price;

s.t. C_Income_From_Electricity_Calculation: vo_income_electricity =
	sum {k in Plants} v_sell_electricity[k] * Plants_Common_data['SELL_PRICE_ELECTRICITY',k];

s.t. C_Income_Total_Calculation: vo_yearly_income = 0
	+ vo_income_heat
	+ vo_income_electricity;

### Constraints: Investment costs

var vo_invest_centralized_chp; # EUR
var vo_invest_decentralized_chp; # EUR
var vo_invest_fermenter; # EUR
var vo_invest_silo; # EUR
var vo_invest_transformer; # EUR
var vo_invest_biogas_pipe_fixed; # EUR
var vo_invest_biogas_pipe_proportional; # EUR
var vo_invest_heat_pipe; # EUR
var vo_investment_cost; # EUR

s.t. C_Invest_Centralized_CHP_Calculation: vo_invest_centralized_chp =
	sum {k in Plants} v_use_centralized_chp[k] * Plants_Common_data['INVESTMENT_COST_CHP',k];

s.t. C_Invest_Decentralized_CHP_Calculation: vo_invest_decentralized_chp =
	sum {k in Plants, l in Locations} v_use_decentralized_chp[k,l] * Plants_Common_data['INVESTMENT_COST_CHP',k];

s.t. C_Invest_Fermenter_Calculation: vo_invest_fermenter =
	sum {k in Plants, m in Fermenter_Mixtures, l in Locations} v_use_fermenter[k,m,l] * Plants_Specific_data['INVESTMENT_COST_FERMENTER',k,m];

s.t. C_Invest_Silo_Calculation: vo_invest_silo =
	Silo_Plate_Investment_Cost * sum {l in Locations} v_use_silo_plate[l];

s.t. C_Invest_Transformer_Calculation: vo_invest_transformer =
	v_use_transformer * Transformer_Cost;

s.t. C_Invest_Biogas_Pipe_Fixed_Calculation: vo_invest_biogas_pipe_fixed =
	sum {p in Pipe_Sections} v_use_biogas_pipe[p] * Biogas_Pipe_Fixed_Investment_Cost[p];

s.t. C_Invest_Biogas_Pipe_Proportional_Calculation: vo_invest_biogas_pipe_proportional =
	sum {p in Pipe_Sections} v_use_biogas_pipe[p] * Biogas_Pipe_Propotional_Investment_Cost * Pipe_Section_Length[p];

s.t. C_Invest_Heat_Pipe_Calculation: vo_invest_heat_pipe =
	sum {p in Pipe_Sections} v_use_heat_pipe[p] * Heat_Pipe_Proportional_Investment_Cost * Pipe_Section_Length[p];

s.t. C_Investment_Total_Calculation: vo_investment_cost = 0
	+ vo_invest_centralized_chp
	+ vo_invest_decentralized_chp
	+ vo_invest_fermenter
	+ vo_invest_silo
	+ vo_invest_transformer
	+ vo_invest_biogas_pipe_fixed
	+ vo_invest_biogas_pipe_proportional
	+ vo_invest_heat_pipe;

### Constraints: Yearly expenses

var vo_substrate_cost; # EUR/y
var vo_operation_centralized_chp; # EUR/y
var vo_operation_decentralized_chp; # EUR/y
var vo_electricity_cost_centralized_chp; # EUR/y
var vo_electricity_cost_decentralized_chp; # EUR/y
var vo_operation_fermenter; # EUR/y
var vo_operation_silo; # EUR/y
var vo_operation_heating_fermenters; # EUR/y
var vo_transport_heat_electricity_cost; # EUR/y
var vo_transport_substrates_fixed; # EUR/y
var vo_transport_substrates_proportional; # EUR/y
var vo_yearly_expenses; # EUR/y

param Substrates_Used_Cost {t in Substrates} := 0
	+ Substrates_Cost_Use_CMIN * Substrates_data['CMIN',t]
	+ Substrates_Cost_Use_CMAX * Substrates_data['CMAX',t];
s.t. C_Substrate_Cost_Calculation: vo_substrate_cost =
	sum {t in Substrates} Substrates_Used_Cost[t] * Substrates_data['DM',t] * sum {s in Suppliers, l in Locations} v_transport_substrates[s,l,t];

s.t. C_Operations_Centralized_CHP_Calculaton: vo_operation_centralized_chp =
	sum {k in Plants} v_use_centralized_chp[k] * Plants_Common_data['OPERATION_COST_CHP',k];

s.t. C_Operations_Decentralized_CHP_Calculation: vo_operation_decentralized_chp =
	sum {k in Plants, l in Locations} v_use_decentralized_chp[k,l] * Plants_Common_data['OPERATION_COST_CHP',k];

s.t. C_Electricity_Cost_Centralized_CHP_Calculation: vo_electricity_cost_centralized_chp =
	sum {k in Plants} v_chp_centralized_hours[k] * Electricity_Purchase_Price *
	(Plants_Common_data['SYSTEM_POWER_ELECTRICITY',k] / 1000) * Plants_Common_data['ELECTRICITY_REQUIREMENT_FACTOR',k];

s.t. C_Electricity_Cost_Decentralized_CHP_Calculation: vo_electricity_cost_decentralized_chp =
	sum {k in Plants, l in Locations} v_chp_decentralized_hours[k,l] * Electricity_Purchase_Price *
	(Plants_Common_data['SYSTEM_POWER_ELECTRICITY',k] / 1000) * Plants_Common_data['ELECTRICITY_REQUIREMENT_FACTOR',k];

s.t. C_Operations_Fermenter_Calculation: vo_operation_fermenter =
	sum {k in Plants, m in Fermenter_Mixtures, l in Locations} v_use_fermenter[k,m,l] * Plants_Common_data['OPERATION_COST_FERMENTER',k];

s.t. C_Operations_Silo_Calculation: vo_operation_silo =
	Silo_Plate_Operating_Cost * sum {l in Locations} v_use_silo_plate[l];

s.t. C_Operations_Heating_For_Fermenters_Calculation: vo_operation_heating_fermenters =
	Fermenter_Heating_Cost * sum {l in Locations} v_fermenter_heating_purchased[l];

s.t. C_Transportation_Heat_Calculation: vo_transport_heat_electricity_cost =
	Electricity_Purchase_Price * Heat_Pipe_Electricity_Need * sum {l in Locations} v_transport_heat[l];

s.t. C_Transportation_Substrates_Fixed_Calculation: vo_transport_substrates_fixed =
	sum {s in Suppliers, l in Locations, t in Substrates} v_transport_substrates[s,l,t] * Transporting_Costs_Fixed[t];

s.t. C_Transportation_Substrates_Proportional_Calculation: vo_transport_substrates_proportional =
	sum {s in Suppliers, l in Locations, t in Substrates} v_transport_substrates[s,l,t] * Transporting_Costs_Proportional[t] * Distance[s,l];

s.t. C_Yearly_Expenses_Total_Calculation: vo_yearly_expenses = 0
	+ vo_substrate_cost
	+ vo_operation_centralized_chp
	+ vo_operation_decentralized_chp
	+ vo_electricity_cost_centralized_chp
	+ vo_electricity_cost_decentralized_chp
	+ vo_operation_fermenter
	+ vo_operation_silo
	+ vo_operation_heating_fermenters
	+ vo_transport_heat_electricity_cost
	+ vo_transport_substrates_fixed
	+ vo_transport_substrates_proportional;

### Objective

var v_profit;

s.t. C_Profit_Calculation:
	v_profit = 0
	+ vo_yearly_income
	- vo_investment_cost / Deprecation
	- vo_yearly_expenses;

maximize Obj: v_profit;

solve;

printf "\nPROFIT = %g\n", v_profit;

printf "\nDECISION VARIABLES:\n";
for {k in Plants: v_use_centralized_chp[k]}
{
	printf "CHP CENTER %s: %d %g\n", k, v_use_centralized_chp[k], v_chp_centralized_hours[k]; 
}
for {k in Plants, l in Locations: v_use_decentralized_chp[k,l]}
{
	printf "CHP AT %s %s: %d %g\n", l, k, v_use_decentralized_chp[k,l], v_chp_decentralized_hours[k,l]; 
}
for {k in Plants, m in Fermenter_Mixtures, l in Locations: v_use_fermenter[k,m,l]}
{
	printf "FERMENTER AT %s %s %s: %d %g\n", l, m, k, v_use_fermenter[k,m,l], v_fermenter_hours[k,m,l];
}
for {l in Locations: v_use_silo_plate[l]}
{
	printf "SILO PLATE AT %s\n", l;
}
printf "TRANSFORMER: %s\n", (if (v_use_transformer) then "YES" else "NO");
for {l in Locations: v_is_biogas_transport[l]}
{
	printf "TRANSPORT BIOGAS FROM %s\n", l;
}
for {l in Locations: v_is_heat_transport[l]}
{
	printf "TRANSPORT HEAT FROM %s\n", l;
}
for {p in Pipe_Sections: v_use_biogas_pipe[p]}
{
	printf "UTILIZE BIOGAS PIPE %s\n", p;
}
for {p in Pipe_Sections: v_use_heat_pipe[p]}
{
	printf "UTILIZE HEAT PIPE %s\n", p;
}

set DisplayValues, dimen 2 := {
	('STROMERLOS',vo_income_electricity),
	('WARMEERLOS',vo_income_heat),
	('SUMME_ERLOSE',vo_yearly_income),
	('BETRIEBKOSTEN_SILOPLATTEN',vo_operation_silo),
	('SUBSTRATKOSTEN',vo_substrate_cost),
	('TRANSPORTKOSTEN',vo_transport_substrates_fixed + vo_transport_substrates_proportional),
	('STROMKOSTEN_1',vo_electricity_cost_decentralized_chp + vo_electricity_cost_centralized_chp + vo_transport_heat_electricity_cost),
	('STROMKOSTEN_2',vo_electricity_cost_decentralized_chp + vo_electricity_cost_centralized_chp),
	('SUMME_BETRIEBSKOSTEN',vo_yearly_expenses),
	('BETRIEBSERGEBNIS_VOR_AFA',vo_yearly_income-vo_yearly_expenses),
	('INVESTIONKOSTEN',vo_investment_cost),
	('BETRIEBSERGEBNIS_NACH_AFA',vo_yearly_income - vo_yearly_expenses - vo_investment_cost / Deprecation),

	('VO_TRANSPORT_HEAT_ELECTRICITY_COST',vo_transport_heat_electricity_cost),
	('V_TRANSPORT_HEAT_TOTAL',sum {l in Locations} v_transport_heat[l]),
	('V_SELL_ELECTRICITY_TOTAL',sum {k in Plants} v_sell_electricity[k]),
	('V_HEAT_LOSS_TOTAL',sum {p in Pipe_Sections} v_heat_loss[p]),
	('V_SELL_HEAT',v_sell_heat),
	('FERMENTER_HEATING_PURCHASED',sum {l in Locations} v_fermenter_heating_purchased[l])
};

printf "\nIMPORTANT VALUES:\n";
for {(name,value) in DisplayValues}
{
	printf "%s = %g\n", name, value;
}

printf "\nSUBSTRATE USAGE:\n";

for {s in Suppliers, t in Substrates}
{
	printf "%2s, %8s: (%7g / %7g) (%d %%)\n", s, t,
		sum {l in Locations} v_transport_substrates[s,l,t],
		Substrate_Available_FM[s,t],
		round(100 * (sum {l in Locations} v_transport_substrates[s,l,t]) / Substrate_Available_FM[s,t]);
}

data;

set Suppliers := L1 L2 L3 L4 L5 L6 L7 L8;

set Individual_Suppliers := A B C D E F G H I J K L M N P Q R S T;

param Individual_Supplier_Group :=
	A	L1
	B	L2
	C	L3
	D	L3
	E	L4
	F	L4
	G	L4
	H	L5
	I	L5
	J	L6
	K	L6
	L	L3
	M	L7
	N	L7
	P	L8
	Q	L8
	R	L2
	S	L8
	T	L8
	;

set Locations := Loc1 Loc2 Loc3;

param Distance:
		Loc1	Loc2	Loc3 := # km
	L1	1.6	3.4	0
	L2	3.3	4.7	4.0
	L3	2.7	4.6	1.2
	L4	1.9	1.4	3.3
	L5	0.3	2.1	2.1
	L6	1.5	2.9	3.0
	L7	3.1	3.0	2.4
	L8	3.8	1.9	3.7
	;

set Pipe_Sections := Pipe1 Pipe2 Pipe3to1;

param Pipe_Section_Length := # km
	Pipe1	1.1
	Pipe2	1.0
	Pipe3to1	1.3
	;

set Pipe_Requirements :=
	Loc1	Pipe1
	Loc2	Pipe2
	Loc3	Pipe1
	Loc3	Pipe3to1
	;

set Substrates := Manure Inter Grass Corn; # Inter: intercrops.

param Substrates_data (tr): # DM: ratio, CH4: m3/t DM, C: EUR/t DM, SPACE: t/ha
		DM	CH4	CMIN	CMAX	YIELD :=
	Manure	0.09	200	5	10	0
	Inter	0.24	300	50	80	2.8
	Grass	0.30	300	50	80	12
	Corn	0.33	340	65	110	15
	;

param Substrates_Availability:
		Manure	Inter	Grass	Corn :=
	L1	4511	370	246	232
	L2	1100	90	60	57
	L3	2089	172	114	107
	L4	1867	154	102	96
	L5	878	72	48	45
	L6	1100	90	60	57
	L7	1756	144	96	91
	L8	2200	180	120	113
	;

# manure was given as raw m3, others are given as DM t
# manure: 9% DM content for volume, 5% DM content for mass (?)
param Substrate_DM_for_Availability_Factor :=
	Manure	0.09 # unit: m3
	Inter	1 # unit: t
	Grass	1 # unit: t
	Corn	1 # unit: t
	;

# Die freie Zusammenstellung einer Substratmischung durch die PNS ist nicht möglich, weil die
# verschiedenen Substrate unterschiedliche Fermentervolumen, Einbringungs- und Rührtechnik
# erfordern und damit variierende Investitionskosten zur Folge haben. Für die Optimierung wurde
# deshalb mit acht unterschiedlichen Substratbeschickungen der Fermenter gearbeitet.

# The free composition of a substrate mixture by the PNS is not possible, because the different
# substrates require different fermenter volumes, feeding and stirring technology and thus result
# in varying investment costs. For the optimization, eight different substrate feeds were
# therefore used in the fermenter.

set Fermenter_Mixtures := Mix1 Mix2 Mix3 Mix4 Mix5 Mix6 Mix7 Mix8;

param Fermenter_Mixtures_Compositions:
		Manure	Inter	Grass	Corn :=
	Mix1	30	0	0	70
	Mix2	30	70	0	0
	Mix3	50	50	0	0
	Mix4	50	20	10	20
	Mix5	75	0	0	25
	Mix6	75	25	0	0
	Mix7	75	15	10	0
	Mix8	100	0	0	0
	;

param Full_Load_Hours := 7800;

param Deprecation := 15;

set Plants := K80 K160 K250 K500;

param Plants_Common_data:
		K80	K160	K250	K500 :=
	SYSTEM_POWER_ELECTRICITY # kW
		80	160	250	500
	SYSTEM_POWER_HEAT # kW
		105	230	300	560
	ELECTRICITY_REQUIREMENT_FACTOR # power times factor gives required amount
		0.06	0.06	0.06	0.06
	SELL_PRICE_ELECTRICITY # EUR/MWh
		205	205	205	185 # CHP, >=30% manure: 185 if <500 kW, 165 if >=500 kW, +20 CHP bonus
	INVESTMENT_COST_CHP # EUR
		110000	200000	250000	430000
	OPERATION_COST_FERMENTER # EUR/y
		18888	23300	29213	46706
	OPERATION_COST_CHP # EUR/y
		27138	33098	37778	51346
	;

param Plants_Specific_data :=
	[HEAT_DEMAND,*,*] (tr): # MWh
		K80	K160	K250	K500 :=
	Mix1	79.7	159.3	248.9	497.9
	Mix2	112.2	224.4	350.6	701.2
	Mix3	134.6	269.3	420.7	841.4
	Mix4	108.5	216.9	338.9	667.8
	Mix5	158.6	317.1	495.5	991.0
	Mix6	194.5	388.9	607.7	1215.4
	Mix7	182.0	364.0	568.7	1137.3
	Mix8	368.6	737.1	1151.7	2303.4
	[INVESTMENT_COST_FERMENTER,*,*] (tr): # EUR
		K80	K160	K250	K500 :=
	Mix1	374892	589477	823606	1221043
	Mix2	411303	683606	850921	1515837
	Mix3	458327	789654	1011043	1748756
	Mix4	411303	683606	850921	1451427
	Mix5	458327	789654	1011043	1768756
	Mix6	460704	780409	1031043	1884821
	Mix7	474704	822409	1075558	1798756
	Mix8	516058	981547	1305558	2603514
	;

param Transformer_Cost := 35000; # EUR

param Electricity_Purchase_Price := 150; # EUR/MWh

param Fermenter_Heating_Cost := 50; # EUR/MWh

param Silo_Plate_Investment_Cost := 150000; # EUR

param Silo_Plate_Operating_Cost := 5850; # EUR/y

param Transporting_Costs_Fixed := # EUR/unit
	Manure	1 # unit: m3
	Inter	2 # unit: t
	Grass	2 # unit: t
	Corn	2 # unit: t
	;

param Transporting_Costs_Proportional := # EUR/unit/km
	Manure	0.25 # unit: m3
	Inter	0.49 # unit: t
	Grass	0.49 # unit: t
	Corn	0.49 # unit: t
	;

param Biogas_Pipe_Propotional_Investment_Cost := 22100; # EUR/km

param Biogas_Pipe_Fixed_Investment_Cost := # EUR
	Pipe1	20000
	Pipe2	20000
	Pipe3to1	60000
	; # 20000 below 2 km, 60000 above 2 km, note that Pipe3to1 implies Pipe1, 2.4 km in total

param Heat_Pipe_Proportional_Investment_Cost := 300000; # EUR/km

param Heat_Pipe_Transfer_Loss := 105; # MWh/km

param Heat_Pipe_Electricity_Need := 0.005; # ratio per heat fed in

param Heat_Sell_Price := 22.5; # EUR/MWh

end;
