library(mvtnorm)

## rho vector to Omega matrix ##
rhoToOmega <- function(rho){
  dim <- (sqrt(8*length(rho)+1)+1)/2;
  Omega <- diag(1/2,dim);
  Omega[lower.tri(Omega)] <- rho;
  Omega <- Omega + t(Omega);
  Omega;
}

## cumulative distibution function of standard AC skew-t distribution
## redefine pst on "sn" ver.1.2-0
psACst <- function (x, zeta, nu, ...){ 
  ok <- !(is.na(x) | (x==Inf) | (x==-Inf))
  y <- x[ok]
  if(abs(zeta) == Inf) {
    z0 <- replace(y, zeta*y < 0, 0)
    p <- pf(z0^2, 1, nu)
    return(if(zeta>0) p else (1-p))
  }  
  fp <- function(v, zeta, nu, t.value) 
     pssn(sqrt(v) * t.value, zeta) * dchisq(v * nu, nu) * nu
  if((round(nu)==nu) && (nu < (8.20 + 3.55* log(log(length(y)+1))))){
    p <- psACst_int(y, zeta, nu)  # "method 4"
  }
  else{
    p <- numeric(length(y))
    upper <- 10 + 50/nu
    intdsst <- function(q) integrate(dsACst, -Inf, q, zeta, nu, ...)$value
    intfp <- function(q) integrate(fp, 0, Inf, zeta, nu, q, ...)$value
    idx2 <- (y<upper)
    idx3 <- !idx2
    p[idx2] <- sapply(y[idx2],intdsst)  # method 2
    p[idx3] <- sapply(y[idx3],intfp)  # method 3
  }
  pr <- rep(NA, length(x))
  pr[x==Inf] <- 1
  pr[x==-Inf] <- 0
  pr[ok] <- as.numeric(p)
  return(pmax(0,pmin(1,pr)))
}

## quantile function of standard AC skew-t distribution
## redefine qst on "sn" ver.1.2-0
qsACst <- function (p, zeta = 0, nu = Inf, tol = 1e-08, maxit = 30, ...){
  if (zeta == Inf) 
    return(sqrt(qf(p, 1, nu)))
  if (zeta == -Inf) 
    return(- sqrt(qf(1 - p, 1, nu)))
  if (zeta == 0)
    return(qt(p, nu))
  na <- is.na(p) | (p < 0) | (p > 1)
  abs.zeta <- abs(zeta)
  if (zeta < 0) 
    p <- (1 - p)
  zero <- (p == 0)
  one <- (p == 1)
  x <- xa <- xb <- xc <- fa <- fb <- fc <- rep(NA, length(p))
  nc <- rep(TRUE, length(p))
  nc[(na | zero | one)] <- FALSE
  fc[!nc] <- 0
  xa[nc] <- qt(p[nc], nu)
  xb[nc] <- sqrt(qf(p[nc], 1, nu))
  fa[nc] <- psACst(xa[nc], abs.zeta, nu, ...) - p[nc]
  fb[nc] <- psACst(xb[nc], abs.zeta, nu, ...) - p[nc]
  regula.falsi <- FALSE
  it <- 0
  while ((sum(nc) > 0) & (it < maxit)) {
    xc[nc] <- if (regula.falsi) 
      xb[nc] - fb[nc] * (xb[nc] - xa[nc])/(fb[nc] - fa[nc])
    else (xb[nc] + xa[nc])/2
    fc[nc] <- psACst(xc[nc], abs.zeta, nu, ...) - p[nc]
    pos <- (fc[nc] > 0)
    xa[nc][!pos] <- xc[nc][!pos]
    fa[nc][!pos] <- fc[nc][!pos]
    xb[nc][pos] <- xc[nc][pos]
    fb[nc][pos] <- fc[nc][pos]
    x[nc] <- xc[nc]
    nc[(abs(fc) < tol)] <- FALSE
    regula.falsi <- !regula.falsi
    it <- it + 1
  }
  x <- replace(x, zero, -Inf)
  x <- replace(x, one, Inf)
  q <- as.numeric(sign(zeta)* x)
  names(q) <- names(p)
  return(q)
}

## cumulative density function of standard AC skew-t distribution
## with positive integer nu (degree of freedom parameter)
## The algorithm is given by Jamalizadeh, Khosravi, and Balakrishnan (2009)
psACst_int <- function (x, zeta=0, nu=Inf){
  if(nu != round(nu) | nu < 1) stop("nu not integer or not positive")
  if(nu == 1) 
    atan(x)/pi + acos(zeta/sqrt((1+zeta^2)*(1+x^2)))/pi
    else { if(nu==2)
      0.5 - atan(zeta)/pi + (0.5 + atan(x*zeta/sqrt(2+x^2))/pi)*x/sqrt(2+x^2)
    else
      (psACst_int(sqrt((nu-2)/nu)*x, zeta, nu-2) + 
        psACst_int(sqrt(nu-1)*zeta*x/sqrt(nu+x^2), 0, nu-1) * x *
        exp(lgamma((nu-1)/2) +(nu/2-1)*log(nu)-0.5*log(pi)-lgamma(nu/2)
        -0.5*(nu-1)*log(nu+x^2)))
    } 
}


## cumulative density function of standard univariate skew-Normal distribution
pssn <- function(x, zeta=0, engine, ...){
  nx <- length(x)
  na <- length(zeta)
  if(missing(engine)) engine <- 
    if(na == 1 & nx > 3 & all(zeta*x > -5)) 
      "T.Owen" else "binorm"
  if(engine == "T.Owen") {
    if(na > 1) stop("engine='T.Owen' not compatible with other arguments")
    p <- pnorm(x) - 2 * T.Owen(x, zeta, ...)
  }
  else{ #  engine="binorm"
    p <- numeric(nx)
    zeta <- cbind(x, zeta)[,2]
    delta <- zeta/sqrt(1+zeta*zeta)
    for(k in seq_len(nx)) {
      if(abs(zeta[k]) == Inf){
       p[k] <- if(zeta[k] > 0)
             2*pnorm(pmax(x[k],0)) - 1
           else
             2*pnorm(pmin(x[k],0))
      }
      else { # SNbook: formula (2.48), p.40
        R <- matrix(c(1, -delta[k], -delta[k], 1), 2, 2)
        p[k] <- 2 * pmvnorm(upper=c(x[k], 0), corr=R, algorithm=TVPACK)
      }
    }
  }
  pmin(1, pmax(0, as.numeric(p)))
}

## quantile function of standard univariate skew-Normal distribution
## using "NR" solver
qssn <- function(p, zeta = 0, tol = 1e-08, ...){
  max.q <- sqrt(qchisq(p,1));
  min.q <- -sqrt(qchisq(1-p,1));
  if(zeta == Inf) return(as.numeric(max.q))
  if(zeta == -Inf) return(as.numeric(min.q))
  na <- is.na(p) | (p < 0) | (p > 1)
  zero <- (p == 0)
  one <- (p == 1)
  p <- replace(p, (na | zero | one), 0.5)

  delta <- zeta/sqrt(1+zeta^2);
  ex <- sqrt(2/pi)*delta;
  vx <- 1-ex^2;
  sx <- 0.5*(4-pi)*(ex^3);
  kx <- 2*(pi-3)*(ex^4);
  g1 <- sx/(vx^(3/2));
  g2 <- kx/(vx^2);
  x <- qnorm(p)
  x <- (x + (x^2 - 1) * g1/6 + x * (x^2 - 3) * g2/24 -
          x * (2 * x^2 - 5) * g1^2/36)
  x <- ex + sqrt(vx) * x
  px <- pssn(x, zeta=zeta, ...);
  max.err <- 1
  while (max.err > tol) {
    x1 <- x - (px - p)/dssn(x,zeta);
    x <- x1
    px <- pssn(x, zeta=zeta, ...)
    max.err <- max(abs(px-p))
    if(is.na(max.err)) stop('failed convergence, try with solver="RFB"')
  }
  x <- replace(x, na, NA)
  x <- replace(x, zero, -Inf)
  x <- replace(x, one, Inf)
  q <- as.numeric(x)
  names(q) <- names(p)
  return(q)
}

## density of standard univariate skew-Normal distribution
dssn <- function(x, zeta){
 pdf <- dnorm(x);
 if(abs(zeta) < Inf)   
   cdf <- pnorm(zeta*x)
 else
   cdf  <- as.numeric(sign(zeta)*x > 0) 
  2*pdf*cdf;
}

## Bivariate normal cumulative distribution function 
## using the auxilary T function of Owen (1956).
## The codes are from "sn" ver.1.2-0.
T.Owen <- function(h, a, jmax=50, cut.point=8){
  T.int <-function(h, a, jmax, cut.point){
     fui <- function(h,i) (h^(2*i))/((2^i)*gamma(i+1)) 
     seriesL <- seriesH <- NULL
     i  <- 0:jmax
     low<- (h <= cut.point)
     hL <- h[low]
     hH <- h[!low]
     L  <- length(hL)
     if (L > 0) {
       b    <- outer(hL, i, fui)
       cumb <- apply(b, 1, cumsum)
       b1   <- exp(-0.5*hL^2) * t(cumb)
       matr <- matrix(1, jmax+1, L) - t(b1)
       jk   <- rep(c(1,-1), jmax)[1:(jmax+1)]/(2*i+1)
       matr <- t(matr*jk) %*%  a^(2*i+1)
       seriesL  <- (atan(a) - as.vector(matr))/(2*pi)
     }
     if (length(hH) > 0)  seriesH <- 
          atan(a)*exp(-0.5*(hH^2)*a/atan(a)) * (1+0.00868*(hH*a)^4)/(2*pi)
     series <- c(seriesL, seriesH)
     id <- c((1:length(h))[low],(1:length(h))[!low]) 
     series[id] <- series  # re-sets in original order
     series
  }
  if(!is.vector(a) | length(a)>1) stop("'a' must be a vector of length 1")
  if(!is.vector(h)) stop("'h' must be a vector")
  aa <- abs(a)    
  ah <- abs(h)
  if(is.na(aa)) stop("parameter 'a' is NA") 
  if(aa==Inf) return(sign(a)*0.5*pnorm(-ah)) # sign(a): 16.07.2007
  if(aa==0)   return(rep(0,length(h)))
  na  <- is.na(h)
  inf <- (ah == Inf)
  ah  <- replace(ah,(na|inf),0)
  if(aa <= 1)
    owen <- T.int(ah,aa,jmax,cut.point)
  else
    owen<- (0.5*pnorm(ah) + pnorm(aa*ah)*(0.5-pnorm(ah)) 
               - T.int(aa*ah,(1/aa),jmax,cut.point))
  owen <- replace(owen,na,NA)
  owen <- replace(owen,inf,0)
  return(owen*sign(a))
}

## density of standard univariate GH skew-t distribution
## redefine dghyp on "ghyp" ver.1.5.6
"dsGHst" <- function(x, gam, nu, logvalue = FALSE){
    if(gam==0) return(dt(x,df=nu,log=logvalue));
    skewness.scaled <- x * gam;
    skewness.norm <- gam^2;
    hnu.min.0.5 <- -nu/2 - 0.5
    interm <- sqrt((nu + x^2) * skewness.norm)
    log.const.top <- nu/2 * log(nu) - hnu.min.0.5 * log(skewness.norm)
    log.const.bottom <- 0.5 * log(2 * pi) + lgamma(nu/2) - (1-nu/2) * log(2)
    log.top <- log(besselK(interm, hnu.min.0.5, expon.scaled = TRUE)) - interm  + skewness.scaled
    log.bottom <- -hnu.min.0.5 * log(interm)
    out <- log.const.top + log.top - log.const.bottom - log.bottom
    if (!logvalue){
        out <- exp(out)
    }
   return(out)
}


## cumulative distibution function of standard GH skew-t distribution
## redefine pghyp on "ghyp" ver.1.5.6
"psGHst" <- function(q, gam, nu, subdivisions = 200,
                    rel.tol = .Machine$double.eps^0.5, abs.tol = rel.tol,
                    lower.tail = TRUE)
{
    if(gam==0){
        if(nu==Inf){
            return(pnorm(q,lower.tail=lower.tail));
        }else{
            return(pt(q,df=nu,lower.tail=lower.tail));
        }
    }
    q.raw <- q
    q.finite <- q.raw[is.finite(q.raw)]
    q.mat <- matrix(q.finite, ncol = 1)

    p.raw <- rep(NA, length(q.raw))

    pdf.args <- list(gam = gam, nu = nu)

    if(lower.tail){
        p.raw[q.raw == -Inf] <- 0
        p.raw[q.raw == Inf] <- 1
        value <- apply(q.mat, MARGIN = 1, FUN = .p.default, pdf = "dsGHst",
            lower = -Inf, pdf.args = pdf.args, subdivisions = subdivisions,
            rel.tol = rel.tol, abs.tol = abs.tol)
    }else{
        p.raw[q.raw == -Inf] <- 1
        p.raw[q.raw == Inf] <- 0
        value <- apply(q.mat, MARGIN = 1, FUN = .p.default, pdf = "dsGHst",
           upper = Inf, pdf.args = pdf.args, subdivisions = subdivisions,
           rel.tol = rel.tol, abs.tol = abs.tol)
    }
    p.raw[is.finite(q.raw)] <- value
    return(as.vector(p.raw))
}

## quantile function of standard GH skew-t distribution
## redefine qghyp on "ghyp" ver.1.5.6
"qsGHst" <- function(p, gam, nu, method = c("integration","splines"),
                    spline.points = 200, subdivisions = 200,
                    root.tol = .Machine$double.eps^0.5,
                    rel.tol = root.tol^1.5, abs.tol = rel.tol)
{
    if(gam==0){
        if(nu==Inf){
            return(qnorm(p));
        }else{
            return(qt(p,df=nu));
        }
    }
    p.raw  <- p

    method <- match.arg(method)

    p.raw[p.raw < 0 | p.raw > 1] <- NaN
    p.raw[p.raw == 1] <- Inf
    p.raw[p.raw == 0] <- -Inf

    ## If only !is.finite quantiles are passed return NA, NaN, Inf, -Inf
    p <- p.raw[is.finite(p.raw)]
    if(length(p) == 0){
        return(p.raw)
    }
    ##   Use Newton's method to find the range of the quantiles
    internal.bisection <- function(nu, gam, p, tol, rel.tol, abs.tol, subdivisions)
    {

        iter <- 0
        range.found <- FALSE

        tvar <- nu/(nu-2);
        if(nu>4){
            step.size <- sqrt(2*gam*gam*tvar*tvar/(nu-4)+tvar);
        }else{
            step.size <- 0.5;
        }
        if(!is.finite(step.size)){
            step.size <- 0.5;
        }

        q.0 <- gam*tvar;

        q.range <- c(q.0 - step.size, q.0 + step.size)

        while(!range.found & iter < 100){
            iter <- iter + 1

            p.range <- psGHst(q = q.range, gam, nu, rel.tol = rel.tol, abs.tol = abs.tol, subdivisions = subdivisions) - p

            if(any(is.na(p.range))){
                warning("Unable to determine interval where the quantiles are in-between.\n",
                        "Perhaps the skewness is too large!")
                return(NA)
            }

            lower <- p.range[1]
            upper <- p.range[2]

            ##      cat("lower: ", lower,";  upper : ", upper, "\n")
            if(upper < 0 & lower < 0){
                q.range[1] <- q.range[2]
                q.range[2] <- q.range[2] + step.size
                next
            }
            if(upper > 0 & lower > 0){
                q.range[2] <- q.range[1]
                q.range[1] <- q.range[1] - step.size
                next
            }
            if(upper > 0 & lower < 0){
                range.found <- TRUE
            }
        }
        if(iter >= 100){
            warning("Unable to determine interval where the quantiles are in-between.\n",
                    "Perhaps the skewness is too large!")
        }

        pdf.args <- list(gam = gam, nu = nu)
        q.root <- .q.default(p, pdf = "dsGHst", pdf.args = pdf.args,
                             interval = q.range, tol = root.tol,
                             p.lower = -Inf, rel.tol = rel.tol, abs.tol = abs.tol,
                             subdivisions = subdivisions)
        return(q.root)
    }
    ##<---------------- end of Newton iteration  ---------------------->

    if(length(p) == 1){
        ## If a single quantile is requested use the newton method anyway
        value <- internal.bisection(nu, gam, p, root.tol, rel.tol, abs.tol, subdivisions)
        p.raw[is.finite(p.raw)] <- as.numeric(value)
        return(p.raw)
    }else if(length(p) == 2){
        ## If two quantiles are requested use the newton method anyway
        value1 <- internal.bisection(nu, gam, p[1], root.tol, rel.tol, abs.tol, subdivisions)
        value2 <- internal.bisection(nu, gam, p[2], root.tol, rel.tol, abs.tol, subdivisions)
        p.raw[is.finite(p.raw)] <- c(value1, value2)
        return(p.raw)
    }else{
        ## If more than two quantiles are requested use the newton method
        ## to find the range where the quantiles can be found.
        q.min <- internal.bisection(nu, gam, min(p), root.tol, rel.tol, abs.tol, subdivisions)
        q.max <- internal.bisection(nu, gam, max(p), root.tol, rel.tol, abs.tol, subdivisions)

        interval <- c(q.min, q.max)

        if(any(is.na(interval))){ # -> Failed to determine bounds for the quantiles
            p.raw[is.finite(p.raw)] <- NA
            return(p.raw)
        }

        ## Extend the interval by 10 percent so that 'uniroot' does not crash
        interval <- c(interval[1] - 0.01 * diff(range(interval)),
                      interval[2] + 0.01 * diff(range(interval)))

        if(method == "integration"){    # Integration method

            pdf.args <- list(gam = gam, nu = nu)
            p <- matrix(p, ncol = 1)
            value <- apply(p, MARGIN = 1, FUN = .q.default, pdf = "dsGHst",
                           pdf.args = pdf.args, interval = interval, tol = root.tol,
                           p.lower = -Inf, rel.tol = rel.tol, abs.tol = abs.tol,
                           subdivisions = subdivisions)
        }else{                          # Splines method
            interval.seq <- seq(min(interval), max(interval), length = spline.points)
            ## Compute the distribution function to be interpolated by splines
            p.interval <- psGHst(q = interval.seq, nu = nu, gam = gam, rel.tol = rel.tol,
                                abs.tol = abs.tol, subdivisions = subdivisions)

            ## Spline function
            spline.distribution.func <- splinefun(interval.seq, p.interval)

            ## root function:   condition: quantile.root.func == 0
            quantile.root.func <- function(x, tmp.p){
                spline.distribution.func(x) - tmp.p
            }

            value <- p

            for(i in 1:length(p)){
                value[i] <- uniroot(quantile.root.func, interval = interval,
                                    tol = root.tol, tmp.p = p[i])$root
            }
        }
        p.raw[is.finite(p.raw)] <- value
        return(p.raw)
    }
}

### <======================================================================>
".p.default" <- function(q, pdf, pdf.args, lower, upper, ...)
{
    if(missing(upper)){
        int.pars <- list(f = pdf, lower = lower, upper = q)
    }else{
        int.pars <- list(f = pdf, lower = q, upper = upper)
    }
    tmp.prob <- try(do.call("integrate", c(pdf.args, int.pars, list(...))))
    if(class(tmp.prob) == "try-error"){
        warning("Failed to determine probability with 'q = ", q,
                "'!\nMessage: ", as.character(tmp.prob), "\n")
        return(NA)
    }else{
        return(tmp.prob$value)
    }
}
### <---------------------------------------------------------------------->


### <======================================================================>
".q.default" <- function(p, pdf, pdf.args, interval, p.lower, ...)
{
    if(p > 0 & p < 1){
        dist.func <- function(x, pdf, p.args, p, p.lower, ...){
            ret.val <- .p.default(q = x, pdf = pdf, pdf.args = p.args,
                                 lower = p.lower, ...)
            return(ret.val - p)
        }
        tmp.quantile <- try(uniroot(dist.func, interval = interval, pdf = pdf,
                                    p.args = pdf.args, p = p,
                                    p.lower = p.lower, ...))

        if(class(tmp.quantile) == "try-error"){
            warning("Failed to determine quantile with 'probs = ", p,
                    "'!\nMessage: ", as.character(tmp.quantile), "\n")
            return(NA)
        }else{
            return(tmp.quantile$root)
        }
    }else if(p == 0){
        return(p.lower)
    }else if(p == 1){
        return(+Inf)
    }else{
        return(NA)
    }
}
### <---------------------------------------------------------------------->

