source("qsstn.R")
library(mvtnorm);

## Numerical lambda_L of bivariate Skew-Normal using mvtnorm
lambdaLSN_Num <- function(u, edelta=0, trho=0){
  rho <- trho*(1-edelta*edelta)+edelta*edelta;
  ezeta <- edelta/sqrt(1-edelta*edelta);
  lx <- qssn(u,ezeta);
  R_L <- rhoToOmega(c(-edelta,-edelta,rho));
  lv <- 2*pmvnorm(upper=c(0,lx,lx),mean=rep(0,3),corr=R_L,algorithm=TVPACK)[1];
  lv/u;
}

# Figure 1: Contour plot of skew-normal copula
dEquiDelta <- function(x,y,trho=0.5){
  ezeta <- edelta/sqrt(1-edelta*edelta);
  cx <- qssn(pnorm(x),ezeta);
  cy <- qssn(pnorm(y),ezeta);
  adjF <- dnorm(x)*dnorm(y)/(dssn(cx,ezeta)*dssn(cy,ezeta));
  rho <- trho + edelta*edelta*(1-trho)
  sqt <- sqrt(1-rho*rho);
  ealpha <- ezeta/sqrt((1+rho)*(1+trho));
  2*dnorm(cx)*dnorm((cy-rho*cx)/sqt)/sqt*pnorm(ealpha*(cx+cy))*adjF;
}
grids <- seq(-3, 3, length=200)
pdf("SNContourPlot.pdf")
par(fig=c(0,0.55,0.4,1))
# Normal
edelta <- 0;
contour(grids,grids,outer(grids,grids,dEquiDelta),labcex=1,xlab="",ylab="",cex.axis=1.1);
mtext(expression(italic(y)[1]),side=1,adj=0.99,padj=-1.5,cex=1.1);
mtext(expression(italic(y)[2]),side=3,adj=0.01,padj=1.5,cex=1.1);
text(.5,-2.7,expression(delta[1] == delta[2]),cex=1.2)
text(1.5,-2.7," = 0",cex=1.2)
par(fig=c(0.45,1,0.4,1),new=TRUE)
# Skew-Normal (edelta=-.8)
edelta <- -0.8;
contour(grids,grids,outer(grids,grids,dEquiDelta),labcex=1,xlab="",ylab="",cex.axis=1.1);
mtext(expression(italic(y)[1]),side=1,adj=0.99,padj=-1.5,cex=1.1);
mtext(expression(italic(y)[2]),side=3,adj=0.01,padj=1.5,cex=1.1);
text(.5,-2.7,expression(delta[1] == delta[2]),cex=1.2)
text(1.75,-2.7," = 0.8",cex=1.2)
dev.off()

## Figure 2: Plot of Tail Asymmmetry in terms of equi-delta with rho
xx <- seq(-.999,.999,length.out=200); ## delta
trhos <- seq(-.8,.8,.2);
lr <- length(trhos);
yy <- matrix(0,nrow=length(xx),ncol=lr);
u <- 0.01;
for(i in 1:length(xx)){
  for(j in 1:length(trhos)){
    yy[i,j] <- log(lambdaLSN_Num(u,edelta=-xx[i],trho=trhos[j])/lambdaLSN_Num(u,edelta=xx[i],trho=trhos[j]));
  }
}
pdf("TailAsymmetryDelta.pdf")
par(fig=c(0,1,0.15,1))
plot(xx,yy[,1],type="l",xlim=c(-1,1),xlab="",ylab="",col=1,lty=1,lwd=2,cex.axis=1.1);
for(j in 2:lr){
  lines(xx,yy[,j],lty=j,col=j,lwd=2);
}
mtext(expression(delta),side=1,adj=1,line=0.1,cex=1.2);
mtext(expression(nu(italic(u))),side=3,adj=0.01,line=-1.5,cex=1.2);
par(xpd=T)
rhotxt <- sprintf("tilde(rho) == %1.1f",trhos);
legend("bottomright",legend=parse(text=rhotxt),lty=c(1:lr),col=c(1:lr),box.lty=0,lwd=2);
text(0,max(yy),parse(text=sprintf("italic(u) == %f",u)),cex=1.2)
dev.off()

## Figure 3: Plot of Tail Asymmmetry in terms of log(u) with equi-delta and trho=0
deltas <- seq(0,.9,.1);
us <- exp(-(1:14))
yy <- matrix(0,nrow=length(us),ncol=length(deltas));
for(i in 1:length(us)){
  for(j in 1:length(deltas)){
    yy[i,j] <- log(lambdaLSN_Num(us[i],edelta=-deltas[j],trho=0)/lambdaLSN_Num(us[i],edelta=deltas[j],trho=0));
  }
}
pdf("TailAsymmetrymlogU.pdf")
# par(mar=c(3,3,4,8),mfrow = c(2, 1))
par(fig=c(0,1,0.3,1))
plot(-log(us),yy[,1],type="l",ylim=c(0,10),xlab="",ylab="",col=1,lty=1,lwd=2,cex.axis=1.1);
for(j in 2:length(deltas)){
  lines(-log(us),yy[,j],col=j,lty=j,lwd=2);
}
mtext(expression(-log(italic(u))),side=1,adj=.95,line=.5,cex=1.2);
mtext(expression(nu (italic(u))),side=3,adj=-.08,line=-2.5,cex=1.2);
par(xpd=T)
deltatxt <- sprintf("delta == %1.1f",deltas);
legend("topleft",legend=parse(text=deltatxt),lty=c(1:length(deltas)),col=c(1:length(deltas)),box.lty=0,lwd=2,cex=1.1);
text(8,9.5,parse(text=sprintf("tilde(rho) == %f",0)),cex=1.2)
dev.off()

## Figure 4: Plot of log of lower tail dependence with u=0.01 in terms of equi-delta with various rhos
xx <- seq(-.999,.999,length.out=200); ## delta
trhos <- seq(.8,-.8,-.2);
lr <- length(trhos);
yy <- matrix(0,nrow=length(xx),ncol=lr);
u <- 0.01;
for(i in 1:length(xx)){
  for(j in 1:lr){
    yy[i,j] <- log(lambdaLSN_Num(u,edelta=xx[i],trho=trhos[j]));
  }
}
pdf("loglambdaLDelta.pdf")
par(fig=c(0,1,0.3,1))
plot(xx,yy[,lr],type="l",xlim=c(-1,1),xlab="",ylab="",col=lr,lwd=2,cex.axis=1.1);
for(j in 1:(lr-1)){
  lines(xx,yy[,j],lty=j,col=j,lwd=2);
}
mtext(expression(delta),side=1,adj=1,line=0.1,cex=1.2);
mtext(expression(paste("log ",italic(lambda)[L](u))),side=3,adj=.02,line=-3.5,cex=1.1);
# par(xpd=T)
rhotxt <- sprintf("tilde(rho) == %1.1f",trhos);
legend("bottomleft", legend=parse(text=rhotxt),lty=c(1:lr),col=c(1:lr),box.lty=0,lwd=2,cex=1.1);
text(0.75,-25,parse(text=sprintf("italic(u) == %f",u)),cex=1.2)
lines(c(-1.1,1.1),c(u,u),lwd=1,lty="dotted",col="grey")
dev.off()

## Figure 5-6: Contour plot of \log \lambda_L(0.01; \delta, \tilde{\rho}) using TVPACK, and the approximations by Fung and Senata (2016) and Li and Joe (2023).

## Asymptotic expression of \lambda_L(u) for the bivariate skew-normal by Fung and Seneta (2016)
lambdaLSN_FS <- function(u, edelta=0, trho=0){
  rho <- trho*(1-edelta*edelta)+edelta*edelta;
  ezeta <- edelta/sqrt(1-edelta*edelta);
  beta <- sqrt((1-trho)/(1+trho));  betasq <- beta*beta;
  if(ezeta>0){
    (u^betasq)*((2*pi*ezeta)^betasq)/(sqrt(pi)*beta*((1+betasq)^2))*((-log(u))^(betasq-0.5))
  }else{
    (u^((1-trho)/(1+trho)))*(1+trho)/2*sqrt((1+trho)/(1-trho))*((-pi*log(u))^(-trho/(1+trho)))
  }
}

## Asymptotic expression of \lambda_L(u) for the bivariate skew-normal by Li and Joe (2023)
lambdaLSN_LJ <- function(u, edelta=0, trho=0){
  rho <- trho*(1-edelta*edelta)+edelta*edelta;
  ezeta <- edelta/sqrt(1-edelta*edelta);
  if(ezeta>0){
    kappa <- 2/(1+trho);
    (1/kappa)*(u^(kappa-1))*((-2*log(u))^(kappa-3/2))
  }else{
    kappa <- 2/(1+rho);
    (1/kappa)*(u^(kappa-1))*((-2*log(u))^(-rho/(1+rho)))
  }
}

trhos <- seq(-.80,.999,length=200)
deltas <- seq(-.999,.999,length=200)
zLJ <- zFS <- zNum <- matrix(0,nrow=length(deltas),ncol=length(trhos))
u <- 0.01
for(i in 1:length(deltas)){
  for(j in 1:length(trhos)){
    zNum[i,j] <- log(lambdaLSN_Num(u,edelta=deltas[i],trho=trhos[j]))
    zFS[i,j] <- log(lambdaLSN_FS(u,edelta=deltas[i],trho=trhos[j]))
    zLJ[i,j] <- log(lambdaLSN_LJ(u,edelta=deltas[i],trho=trhos[j]))
  }
}
zDiffFS <- zFS - zNum
zDiffLJ <- zLJ - zNum

## Figure 5: Contour plot of \log \lambda_L(0.01; \delta, \tilde{\rho}) with the algorithm TVPACK
pdf("loglambdaL.pdf")
par(fig=c(0,1,0.3,1))
alevels=c(0,.5,1,1.5,2.5,3.5,5,7,10,20)
hclcol=hcl.colors(length(alevels),"YlOrRd",rev=TRUE)
image(deltas,trhos,abs(zNum),col=hclcol,breaks=c(alevels,100),xlab="",ylab="")
contour(deltas,trhos,zNum,levels=-alevels,labcex=1,cex.axis=1.1,add=T);
mtext(expression(delta),side=1,adj=.99,line=0.1,cex=1.2);
mtext(expression(tilde(rho)),side=3,adj=-.03,padj=2,cex=1.2);
dev.off()

## Figure 6 : Contour plot of the difference of \log \lambda_L(0.01; \delta, \tilde{\rho}) between the algorithm TVPACK and the preceding results
## Figure 6 (top): Compared with the approxmation by Fung and Seneta (2016)
pdf("dFSloglambdaL.pdf")
par(fig=c(0,1,0.3,1))
adlevels=c(0,.1,.2,.5,1,2,5,10)
dlevels=c(adlevels[-1],0,-adlevels[-1])
hclcol=hcl.colors(length(adlevels),"YlOrRd",rev=TRUE)
dbreaks=c(adlevels,100)
image(deltas,trhos,abs(zDiffFS),col=hclcol,breaks=dbreaks,xlab="",ylab="");
contour(deltas,trhos,zDiffFS,levels=dlevels,labcex=1,cex.axis=1.1,add=T);
mtext(expression(delta),side=1,adj=.99,line=0.1,cex=1.2)
mtext(expression(tilde(rho)),side=3,adj=-.03,padj=2,cex=1.2)
dev.off()

## Figure 6 (bottom): Compared with the approxmation by Li and Joe (2023)
pdf("dLJloglambdaL.pdf")
par(fig=c(0,1,0.3,1))
adlevels=c(0,.1,.2,.5,1,2,5,10)
dlevels=c(adlevels[-1],0,-adlevels[-1])
hclcol=hcl.colors(length(adlevels),"YlOrRd",rev=TRUE)
dbreaks=c(adlevels,100)
image(deltas,trhos,abs(zDiffLJ),col=hclcol,breaks=dbreaks,xlab="",ylab="");
contour(deltas,trhos,zDiffLJ,levels=dlevels,labcex=1,cex.axis=1.1,add=T);
mtext(expression(delta),side=1,adj=.99,line=0.1,cex=1.2)
mtext(expression(tilde(rho)),side=3,adj=-.03,padj=2,cex=1.2)
dev.off()