% NMT.m
%
% Fits measured diatomic specta using line strength data files constructed for selected transitions. 
% The program is designed using a previous FORTRAN/Windows7 implementation including private communications 
% with James O. Hornkohl and David M Surmick.
% 
% inputs: WL_exp - exerimental wavelengths (n x 1 array)
%         Dat    - experimental spextrum (n x 1 array)
%         FWHM   - measured spectral resolution, seed for varried FWHM or
%                  fixed
%         T      - temperature seed for fitting
%         tol    - tolerance of Nelder-Mead fit
%         x      - name of line strength file for calculating theory spectra
%         FIT    - enter 1 for fitting linear offset and temperature
%                  enter 2 for fitting linear offset, temperature, and FWHM
% 
% outputs: profile - matrix containing experimental wavelengths, measured
%                    spectrum, fitted spectrum, fitted baseline offset 
%                    (n x 4 matrix)
%          vals    - array containing fitted paramters (3x1 or 4x1 array),
%                    temperature is always last entry
% 
% sub-functions: FitSpec, FitSpec1, SynthSpec
% 
% Example call: [I,v]=NMT(x,y1,0.15,3000,1e-8,'OH-LSF.txt',2);
%
% David M. Surmick, 04-28-2016, edited by Christian G Parigger 10-22-2022

function [profile,vals] = NMT (WL_exp,Dat,FWHM,T,tol,x,FIT)
tic % start code timer

% global variables
global bFac gFac WLk Tuk TuMin Sk n0 nSpec fwhm delWL temp wl_max; 

% constants in MKS units (Boltzmann factor bfac in cgs units)
h=6.62606957e-34; c=2.99792458e8; kb=1.3806488e-23; bFac=(100*h*c)/kb; gFac=2*sqrt(log(2));

%load experimental data, here an OH spectrum 100 microsecond time delay in air breakdown.
load OH100micros.dat; Dat=OH100micros(:,2); WL_exp=OH100micros(:,1); nSpec=length(Dat);

% input paramters  
T=2000; FWHM=0.3; x='OH-LSF.txt'; temp=T; fwhm=FWHM; wl_min=min(WL_exp); wl_max=max(WL_exp); delWL=(wl_max-wl_min)/(nSpec);

% read MatLab LSF file
[p]=load(x); WN=p(:,1); Tu=p(:,2); S=p(:,3);

% convert vacuum wavenumber to air wavelength
a0=2.72643e-4; a1=1.2288; a2=3.555e4; r=1+a0+(a1./(WN.*WN))+(a2./(WN.*WN.*WN.*WN)); WL=1e7./(r.*WN);

% get LSF table wavelengths in experimental range
A=find(WL>wl_min & WL<wl_max); WLk=WL(A);

% get Term Values and LineStrengths at WLk
Sk=S(A); Tuk=Tu(A); TuMin=min(Tuk);

% get exerpimenal wavelength positions that most closely matches line strength table wavelengths 
n0=zeros(length(WLk),1); for i=1:length(WLk);  [~,n0(i)]=min(abs(WL_exp-WLk(i))); end;

% normalize data
%Dat=Dat/max(Dat);

% Fitting with Nelder-Mead parameters including two cases options
tol=1.e-6; FIT=2; options=optimset('TolX',tol,'MaxIter',1e8,'MaxFunEvals',1e8); 
switch FIT
    case 1 % fit offset, temperature
        theta=ones(3,1);
        theta(3)=T; % temperature seed
        vals=fminsearch(@(x) FitSpec(x,WL_exp,Dat),theta,options);
        bkg=vals(1)+vals(2)*WL_exp; % calculate fitted offset
        [I,bkg1]=SynthSpec(WL_exp,vals(3),FWHM,Dat,bkg); % calculate fit
    case 2 % fit offset, fwhm, temperature
        theta=ones(4,1);
        theta(3)=FWHM; % fwhm seed
        theta(4)=T; % temperature seed
        vals=fminsearch(@(x) FitSpec1(x,WL_exp,Dat),theta,options);
        bkg=vals(1)+vals(2)*WL_exp; % calculate fitted offset
        [I,bkg1]=SynthSpec(WL_exp,vals(4),vals(3),Dat,bkg); % calculate fit
end

% output profille array
%profile=[WL_exp Dat I bkg1];

% Visualize Fit
fname=regexprep(x,'-LSF.txt','-fit:');
figure
switch FIT
    case 1
        plot(WL_exp,Dat,'o',WL_exp,I,WL_exp,bkg1,'LineWidth',1.5)
        legend('experiment','fit','base line')
        set(gca,'FontWeight','bold','FontSize',16,'TickLength',[0.02, 0.02]); 
        val3=round(vals(3),3, 'significant')
        title([num2str(fname),'T=',num2str(vals(3)),'K ,FWHM=',num2str(FWHM),'nm'])
        xlabel('wavelength (nm)')
        ylabel('intensity (a.u.)')
    case 2
        plot(WL_exp,Dat,'o',WL_exp,I,WL_exp,bkg1,'LineWidth',1.5)
        legend('experiment','fit','base line')
        set(gca,'FontWeight','bold','FontSize',20,'TickLength',[0.02, 0.02]); 
        round(vals(4),3,'significant'); round(vals(3),2,'significant');
        val4=round(vals(4),3, 'significant'); val3=round(vals(3),2, 'significant');
        title([num2str(fname),' T=',num2str(val4),' K, FWHM=',num2str(val3),' nm'])
        xlabel('wavelength (nm)','Fontsize',24,'FontWeight','bold')
        ylabel('intensity (a.u.)','Fontsize',24,'FontWeight','bold')
end

toc % end code timer

end % main function

% temperature, offset fit function
function [err] = FitSpec (p,WL_exp,Dat); 
global fwhm;
bkg=p(1)+p(2)*WL_exp; [F,~]=SynthSpec(WL_exp,p(3),fwhm,Dat,bkg); c=F\Dat; z=F*c; err=norm(z-Dat);
end % fit spec 

% temperature, fwhm, offset fit function
function [err] = FitSpec1 (p,WL_exp,Dat);
bkg=p(1)+p(2)*WL_exp; [F,~]=SynthSpec(WL_exp,p(4),p(3),Dat,bkg); c=F\Dat; z=F*c; err=norm(z-Dat); 
end % fit spec 1

% calculate synthetic spectrum for fit
function [I1,bkg1] = SynthSpec (WL_exp,T,FWHM,Dat,bkg);
global bFac gFac WLk Tuk TuMin Sk n0 nSpec delWL wl_max; 
FWHMk=(FWHM*WLk)/wl_max; % wavelength dependent FWHM

% Calculate Peak Intensities 
peak=-4*log(WLk)+log(Sk)-(bFac/T)*(Tuk-TuMin); peak_k=exp(peak);

% calculate synthetic spectrum
I=zeros(nSpec,1); % initialize synthetic spectrum output
for i=1:length(WLk); deln=round(2.5*FWHMk(i)/delWL); nMin=n0(i)-deln;
    if nMin < 1; nMin=1; end;
    nMax=n0(i)+deln;
    if nMax > nSpec; nMax=nSpec; end;
    for j=nMin:nMax; u=abs(gFac*(WLk(i)-WL_exp(j))/FWHMk(i)); I(j)=I(j)+peak_k(i)*exp(-u*u); end; 
end % synthetic spectrum loop

% normailze data to measured spectrum
I=I/max(I); I=I+bkg; sxy=sum(Dat.*I); syy=sum(I.*I); nf= sxy/syy; I1=I*nf; bkg1=bkg*nf;
end % SynthSpec