% BESP.m
%
% Calculates diatomic specta using line strength data files constructed for selected transitions. 
% The program is designed using a previous FORTRAN/Windows7 implementation including private communications 
% with James O. Hornkohl and David M Surmick.
% 
% David M. Surmick, 04-27-2016; edited by Christian G. Parigger 10-22-2022.

% input paramters, output: WL_exp (N-1 x 1 array), I (intensity)
wl_min=300; wl_max=325; T=3390; FWHM=0.35; N=1001; norm=1; x='OH-LSF.txt';

% generate wavelengths/wavelength-bins for computation akin to an experiment
nSpec=N-1; delWL=(wl_max-wl_min)/(nSpec); WL_exp=linspace(wl_min,wl_max,nSpec); WL_exp=WL_exp';

% constants in MKS units (Boltzmann factor bfac in cgs units)
h=6.62606957e-34; c=2.99792458e8; kb=1.3806488e-23; bFac=(100*h*c)/kb; gFac=2*sqrt(log(2));

% read line strength file
[p]=load(x); WN=p(:,1); Tu=p(:,2); S=p(:,3);

% convert vacuum wavenumber to air wavelength
a0=2.72643e-4; a1=1.2288; a2=3.555e4; r=1+a0+(a1./(WN.*WN))+(a2./(WN.*WN.*WN.*WN)); WL=1e7./(r.*WN);

% get LSF table wavelengths that most closely match the wavelength-bins
A=find(WL>wl_min & WL<wl_max); WLk=WL(A);
    
% get term values and line strengths at WLk in the range wl_min to wl_max
Sk=S(A); Tuk=Tu(A); TuMin=min(Tuk);

% calculate peak intensities and initialize peak_k calculation 
peak=-4*log(WLk)+log(Sk)-(bFac/T)*(Tuk-TuMin); peak_k=zeros(nSpec,1); peakMax=-1;
for i=1:length(peak);  
    if peak(i) > peakMax; peakMax=peak(i); end; 
    if peak(i) ~= 0; peak_k(i)=peak(i)-peakMax; end;
end; peak_k=exp(peak_k);

% get wavelength-bin positions that most closely matches line strength table wavelengths 
n0=zeros(length(WLk),1); for i=1:length(WLk); [~,n0(i)]=min(abs(WL_exp-WLk(i))); end;

% calculate spectrum using Gaussian profiles for peaks, and for wavelength dependent FWHM
I=zeros(nSpec,1); FWHMk=(FWHM*WLk)/wl_max; 
for i=1:length(WLk); deln=round(FWHM/delWL,0); nMin=n0(i)-deln;
    if nMin < 1; nMin=1; end; nMax=n0(i)+deln;
    if nMax > nSpec; nMax=nSpec; end;
    for j=nMin:nMax;  u=abs(gFac*(WL_exp(j)-WLk(i))/FWHMk(i));
        if u <=9.21;  I(j)=I(j)+peak_k(i)*exp(-u*u); end;
    end; 
end; I=norm*I/max(I);

%Display graphical output
figure; plot(WL_exp,I,'LineWidth',1.5); set(gca,'FontWeight','bold','FontSize',20,'TickLength',[0.02, 0.02]); 
LimitsX=xlim; LimitsY=ylim; title('   ','HorizontalAlignment','left','Position', [LimitsX(1)-4, LimitsY(2)]);
xlabel('wavelength (nm)','FontSize',24,'FontWeight','bold');
ylabel('intensity (a.u.)','FontSize',24,'FontWeight','bold');
