function [X,Y,u] = complex_gray_scott(L,N,Du,Dv,alpha,beta,A,tend,dt,pars)

% pars: structure with Gray-Scott model parameters, such as:
% pars.F = 0.03;          % Feed rate, F
% pars.kappa = 0.061;     % Depletion rate, kappa

% Mesh and initial condition
dx = L/N; [X,Y] = meshgrid(-L/2+(0:N-1)*dx+dx/2);  % Mesh [-L/2,L/2]
I0 = find((abs(X(:))<0.04).*(abs(Y(:))<0.04));     % Interior square
u = ones(N);  u(I0) = 0.5;                       % Initial condition
v = zeros(N); v(I0) = 0.25;                      % Initial condition

% Definition of spectral operators (diffusion)
[kx,ky] = meshgrid((0:N-1)*pi/L);                   % 2D wavenumbers
lm = kx.^2+ky.^2;                               % Standard Laplacian
lm = lm.^(alpha/2).*(1+A*sin(beta/2*log(lm)));         % Eigenvalues
lm(1) = 0;                                         % (by definition)
eAu  = exp(-Du*lm*dt);               % Full step in diffusion (in u)
eAu2 = exp(-Du*lm*dt/2);             % Half step in diffusion (in u)
eAv  = exp(-Dv*lm*dt);               % Full step in diffusion (in v)
eAv2 = exp(-Dv*lm*dt/2);             % Half step in diffusion (in v)

% Time evolution
nsteps = round(tend/dt);                      % Number of time steps
nplot = round(50/dt);             % Visualise at specified intervals
u = idctn(eAu2.*dctn(u));      % First half step in diffusion (in u)
v = idctn(eAv2.*dctn(v));      % First half step in diffusion (in v)
for istep = 1:nsteps
    % Reaction term (2nd order modified Euler)
    % 1st state
    uv2 = u.*v.*v;
    fu = -uv2 + pars.F*(1-u);
    fv =  uv2 - (pars.F+pars.kappa)*v;
    u1 = u+dt/2*fu;
    v1 = v+dt/2*fv;
    % 2nd state
    uv2 = u1.*v1.*v1;
    fu = -uv2 + pars.F*(1-u1);
    fv =  uv2 - (pars.F+pars.kappa)*v1;
    u = u+dt*fu;
    v = v+dt*fv;
    % Diffusion term
    if mod(istep,nplot)==0
        u = idctn(eAu2.*dctn(u));    % Half step in diffusion (in u)
        v = idctn(eAv2.*dctn(v));    % Half step in diffusion (in v)

        % Visualisation
        pcolor(X,Y,u); shading interp; axis image;   % Visualisation
        title(['t = ' num2str(istep*dt)]);           % Annotate plot
        caxis([0 1]); drawnow;                % Visualization limits

        u = idctn(eAu2.*dctn(u));    % Half step in diffusion (in u)
        v = idctn(eAv2.*dctn(v));    % Half step in diffusion (in v)
    else
        u = idctn(eAu.*dctn(u));     % Full step in diffusion (in u)
        v = idctn(eAv.*dctn(v));     % Full step in diffusion (in v)
    end
end
