function [X,Y,Z,u] = complex_FHN_reentry3D(L,N,D,alpha,beta,A,tend,dt,pars)

% pars: structure with FHN model parameters, such as:
% pars.a = 0.1;           % Model parameter a
% pars.epsilon = 0.01;    % Model parameter epsilon
% pars.b = 0.5;           % Model parameter b
% pars.gamma = 1.0;       % Model parameter gamma
% pars.delta = 0.0;       % Model parameter delta

% Mesh and initial condition
dx = L/N; x = (0:N-1)*dx+dx/2;                   % x & y coordinates
dz = (L/2)/(N/2); z = (0:N/2-1)*dz+dz/2;         % z coordinate
[X,Y,Z] = meshgrid(x,x,z);                       % Mesh
u = zeros(N,N,N/2); u(1:fix(N/2),1:fix(N/3),1:fix(N/4)) = 1;
v = zeros(N,N,N/2); v(1:fix(N/2),fix(N/3)+1:N,1:fix(N/4)) = 0.1;
                    v(:,:,fix(N/4)+1:fix(N/2)) = 0.1;

% Definition of spectral operators (diffusion)
kx = (0:N-1)*pi/L;                          % 1D wavenumbers (x & y)
kz = (0:N/2-1)*pi/(L/2);                        % 1D wavenumbers (z)
[kx,ky,kz] = meshgrid(kx,kx,kz);                    % 3D wavenumbers
lm = kx.^2+ky.^2+kz.^2;                         % Standard Laplacian
lm = lm.^(alpha/2).*(1+A*sin(beta/2*log(lm)));         % Eigenvalues
lm(1) = 0;                                         % (by definition)
Dlm = D*lm;                            % Eigenvalues times diffusion
eA = exp(-Dlm*dt);                          % Full step in diffusion

% Time evolution
nsteps = round(tend/dt);                      % Number of time steps
nplot = round(5/dt);              % Visualise at specified intervals
for istep = 1:nsteps
    c = (pars.b*u-pars.delta)/pars.gamma;
    u_hat = dctn(u); u0_hat = u_hat(1);
    fu = u.*(1-u).*(u-pars.a)-v;
    fu_hat = dctn(fu); f0_hat = fu_hat(1);
    u_hat = fu_hat./Dlm-(fu_hat./Dlm-u_hat).*eA;
    u_hat(1) = u0_hat+f0_hat*dt;
    u = idctn(u_hat);
    v = c-(c-v).*exp(-pars.epsilon*pars.gamma*dt);
    % Visualisation
    if mod(istep,nplot)==0
        clf; isosurface(X,Y,Z,u,0.5);                % Visualisation
        axis([0 1 0 1 0 0.5]); daspect([1 1 1]);      % Aspect ratio
        title(['t = ' num2str(istep*dt)]);           % Annotate plot
        caxis([0 1]); drawnow;                % Visualisation limits
    end
end
