function [X,Y,u] = complex_FHN_reentry(L,N,D,alpha,beta,A,tend,dt,pars)

% pars: structure with FHN model parameters, such as:
% pars.a = 0.1;           % Model parameter a
% pars.epsilon = 0.01;    % Model parameter epsilon
% pars.b = 0.5;           % Model parameter b
% pars.gamma = 1.0;       % Model parameter gamma
% pars.delta = 0.0;       % Model parameter delta

% Mesh and initial condition
dx = L/N; x = (0:N-1)*dx+dx/2; [X,Y] = meshgrid(x,x);   % Mesh [0,L]
u = zeros(N); u(1:fix(N/2),1:fix(N/3)) = 1;      % Initial condition
v = zeros(N); v(1:fix(N/2),fix(N/3)+1:N) = 0.1;  % Initial condition

% Definition of spectral operators (diffusion)
[kx,ky] = meshgrid((0:N-1)*pi/L);                   % 2D wavenumbers
lm = kx.^2+ky.^2;                               % Standard Laplacian
lm = lm.^(alpha/2).*(1+A*sin(beta/2*log(lm)));         % Eigenvalues
lm(1) = 0;                                         % (by definition)
Dlm = D*lm;                            % Eigenvalues times diffusion
eA = exp(-Dlm*dt);                              % Exponential matrix

% Time evolution
nsteps = round(tend/dt);                      % Number of time steps
nplot = round(5/dt);              % Visualise at specified intervals
for istep = 1:nsteps
    c = (pars.b*u-pars.delta)/pars.gamma;
    u_hat = dctn(u); u0_hat = u_hat(1);
    fu = u.*(1-u).*(u-pars.a)-v;
    fu_hat = dctn(fu); f0_hat = fu_hat(1);
    u_hat = fu_hat./Dlm-(fu_hat./Dlm-u_hat).*eA;
    u_hat(1) = u0_hat+f0_hat*dt;
    u = idctn(u_hat);
    v = c-(c-v).*exp(-pars.epsilon*pars.gamma*dt);
    % Visualisation
    if mod(istep,nplot)==0
        pcolor(X,Y,u); shading interp; axis image;   % Visualisation
        title(['t = ' num2str(istep*dt)]);           % Annotate plot
        caxis([0 1]); drawnow;                % Visualisation limits
    end
end
