function [X,Y,u] = complex_allen_cahn(L,N,D,alpha,beta,A,tend,dt)

% Mesh and initial condition
dx = L/N; [X,Y] = meshgrid(-L/2+(0:N-1)*dx+dx/2);  % Mesh [-L/2,L/2]
[TH,R] = cart2pol(X,Y);                          % Polar coordinates
u0 = 0.65+0.35*cos(4*TH)-R;                       % Implicit contour
in = u0(:) >= 0;                                     % Points inside
u = -ones(N); u(in) = 1;                         % Initial condition

% Definition of spectral operators (diffusion)
[kx,ky] = meshgrid((0:N-1)*pi/L);                   % 2D wavenumbers
lm = kx.^2+ky.^2;                               % Standard Laplacian
lm = lm.^(alpha/2).*(1+A*sin(beta/2*log(lm)));         % Eigenvalues
lm(1) = 0;                                         % (by definition)
Dlm = D*lm;                            % Eigenvalues times diffusion
eA = exp(-Dlm*dt);                              % Exponential matrix

% Time evolution
nsteps = round(tend/dt);                      % Number of time steps
nplot = round(5/dt);              % Visualise at specified intervals
for istep = 1:nsteps
    u_hat = dctn(u); u0_hat = u_hat(1);
    f_hat = dctn(u-u.^3); f0_hat = f_hat(1);
    u_hat = f_hat./Dlm-(f_hat./Dlm-u_hat).*eA;
    u_hat(1) = u0_hat+f0_hat*dt;
    u = idctn(u_hat);
    % Visualisation
    if mod(istep,nplot)==0
        pcolor(X,Y,u); shading interp; axis image;   % Visualisation
        title(['t = ' num2str(istep*dt)]);           % Annotate plot
        caxis([-1 1]); drawnow;               % Visualisation limits
    end
end
