%% Examples of fitting LBF representations to the NaCl data

% Data reported in Brown(1999).  Sections of this script need to be run separately since the
% same variable for the representation is used for different fits and each section generates a plot of the results.

%start with a clean environment
clear
clear functions

%load example data
load NaCldata
% use structure "data" as temporary location of information for linear fitting
data.PV=NaCl.PV;
%switch volumes to densities
data.PV(:,2)=data.PV(:,2).^-1; 
data.Vo=NaCl.Vo^-1;
data.K=[NaCl.K(:,4).^-1 NaCl.K(:,2)];
% set Vflg to zero if densities are the working unit
data.Vflg=0; % tell functions that the data are in density units

%% First example using a single interval to undertake "Global" polynomial fitting

% the structure options contains details of how to fit the data
clear options
options.K_weight=1e-2; %weight of ftting the bulk modulus relative to PV data
% the strain metric is "arbitrary" - any can be made to work.  
options.strainflg='Vinet';
% try any of following:
% Choices are:
      options.strainflg='volume';
%       options.strainflg='comp';
%        options.strainflg='log';
%       options.strainflg='Eulerian';
 %     options.strainflg='Vinet';
% Knots specified in density units
options.knt=[ 1.7   8];

% specify the order of fit
options.k=6;  % equivalent to standard 4th order since spline order is one more than degree of polynomials used

% one stop shopping for a linear solution
LBF=lin_F_fit(data,options);
%
%plot results
range=[1.8 5]; % range in density for the plotting calculations 
Plims=[-5 100]; % range in pressure for the plots 
mk_plots(LBF,NaCl,range,Plims); % a function that handles making plots

%% Try a LBF with 3 intervals that extends from negative pressure to ~100 GPa
% start fresh in setting the options:
clear options
options.K_weight=4e0; %weight of ftting the bulk modulus relative to PV data
options.strainflg='Vinet';
% if appropriate knot locations are known in strain units
% options.kntflg=1;
% options.knt=[ -0.1 -0.04  0.1  0.6];  %vinet
options.knt=[1.7 2 2.2 3 8];


% set regularization (chose points in density where the derivative is to be made small)  
options.Reg=[  3.5 3.7 4.2 4.4 5 6 7 7.7 ]; %locations where derivative is to be made small
options.drv=4; %make 4th derivative small
options.lam=1e-2; % how hard should the regularization be?
options.k=6;% specify the order of fit
% one stop shopping for a linear solution
LBF=lin_F_fit(data,options);

%plot results
range=[1.8 5]; % range in density for plotting
Plims=[-5 100];
mk_plots(LBF,NaCl,range,Plims);

%% Try non-linear optimization of previous result to improve high pressure behavior:
% enter limiting dK/dP at high pressure (high density)
data.Kp=[  5   2.5 ;  7.8  2.5];
options.Kp_weight=3e0; % chose a weight for the dK/dP (how much emphasis on these values)
options.lam=4e-1;
Plims=[-5 100];
LBF_nl=nlin_F_Fitter(LBF,data,options);
% examine results
mk_plots(LBF_nl,NaCl,range,Plims);
