%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Project name: 
%
% Simulation of a septic tank with a sidestream slag filter
% mixing of a septic tank effluent with a slag filter effluent at a fixed
% recirculation ratio
% PHREEQC function
%
% PHREEQC datablock: SOLUTION, REACTION, EQUILIBRIUM_PHASES and MIX
% 
% Code written by Dominique Claveau-Mallet
% Creation: 2018-11-19
% Updated: 2019-10-25
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Description of input data
%
% numerical
% parm.database_path: full path of the database used in PHREEQC simulation
%    example: 'C:\Program Files (x86)\USGS\Phreeqc Interactive 3.3.8-11728\database\phreeqc.dat'
%
% influent properties
% parm.inf: list of reactant and concentration (M) 
%    example: {'CaCl2' 5e-6; 'MnSO4' 2e-4; 'NaHCO3' 1e-6}
% parm.inf_equ: list of phases in equilibrium with the influent. Phase
% followed by target saturation index (or log partial pressure for gas) 
% and initial amount of solid phase.
%    example: {'Calcite' 0 10; 'Hydroxyapatite' 0 0; 'CO2(g)' -3.5 10}%
%
% solubility products
% model.KspCAL: calcite solubility constant (in log)
% model.KspHAP_HO: homogeneous hydroxyapatite solubility constant (in log)
%
% septic tank 
% model.CO2: CO2 input in the septic tank from biological activity (mg/L)
%
% slag filter
% model.CaO: CaO added in the REACTION block (mol/L) to represent slag treatment
% parm.rec: recirculation ratio
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Description of output data
%
% F: raw PHREEQC output file
% output: pH, o-PO4 (mg P/L), Ca (mg/L), DIC (mg C/L), alkalinity (mg
% CaCO3/L) at the septic tank influent, slag filter effluent and septic
% tank effluent
%             
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [F,output] = PHREEQCfct_septictankwithslag(parm,model)

phreeqc = actxserver('IPhreeqcCOM.Object');
phreeqc.LoadDatabase(sprintf('%s',parm.database_path)); % retrieve database
phreeqc.ClearAccumulatedLines();

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% PHREEQC Code

% Database definitions

phreeqc.AccumulateLine('PHASES')

phreeqc.AccumulateLine('Calcite')
phreeqc.AccumulateLine('CaCO3 = CO3-2 + Ca+2')
phreeqc.AccumulateLine(sprintf('log_k	%.16g',model.KspCAL))

phreeqc.AccumulateLine('Hydroxyapatite')
phreeqc.AccumulateLine('Ca5(PO4)3OH = OH- +3 PO4-3 + 5 Ca+2')
phreeqc.AccumulateLine(sprintf('log_k	%.16g',model.KspHAP_HO))

% Calculations - basic parameters

phreeqc.AccumulateLine('CALCULATE_VALUES')

phreeqc.AccumulateLine('o-PO4_mgP_L')
phreeqc.AccumulateLine('-start')
phreeqc.AccumulateLine('10	a = TOT("P")*1000*31')
phreeqc.AccumulateLine('20	SAVE a')
phreeqc.AccumulateLine('-end')
phreeqc.AccumulateLine('Ca_mg_L')
phreeqc.AccumulateLine('-start')
phreeqc.AccumulateLine('10	a = TOT("Ca")*1000*40')
phreeqc.AccumulateLine('20	SAVE a')
phreeqc.AccumulateLine('-end')
phreeqc.AccumulateLine('DIC_mg_L')
phreeqc.AccumulateLine('-start')
phreeqc.AccumulateLine('10	a = TOT("C(4)")*1000*12')
phreeqc.AccumulateLine('20	SAVE a')
phreeqc.AccumulateLine('-end')
phreeqc.AccumulateLine('Alk_mg_CaCO3/L')
phreeqc.AccumulateLine('-start')
phreeqc.AccumulateLine('10	a = Alk*50000')
phreeqc.AccumulateLine('20	SAVE a')
phreeqc.AccumulateLine('-end')

% Data extraction

phreeqc.AccumulateLine('SELECTED_OUTPUT')
phreeqc.AccumulateLine('-reset	false')
phreeqc.AccumulateLine('-pH')
phreeqc.AccumulateLine('-calculate_values o-PO4_mgP_L Ca_mg_L DIC_mg_L Alk_mg_CaCO3/L')

% Simulation of the influent

phreeqc.AccumulateLine('SOLUTION 1')
phreeqc.AccumulateLine('pH 7')
phreeqc.AccumulateLine('REACTION 1') % preparation of initial solution
for i = 1:length(parm.inf(:,1))
    phreeqc.AccumulateLine(sprintf('%s  %.16d ', parm.inf{i,1}, parm.inf{i,2}))
end
phreeqc.AccumulateLine('EQUILIBRIUM_PHASES 1') % equilibrium state of the initial solution prior to the test
for i = 1:length(parm.inf_equ(:,1))
    phreeqc.AccumulateLine(sprintf('%s  %.16d %.16d', parm.inf_equ{i,1}, parm.inf_equ{i,2}, parm.inf_equ{i,3}))
end
phreeqc.AccumulateLine('SAVE solution 1')
phreeqc.AccumulateLine('End')

% Simulation of the slag filter effluent

phreeqc.AccumulateLine(sprintf('Use solution 1'))
phreeqc.AccumulateLine('REACTION')
phreeqc.AccumulateLine(sprintf('CaO	%.16g',model.CaO))
phreeqc.AccumulateLine(sprintf('CaCl2	%.16g',0.4*model.CaO))
phreeqc.AccumulateLine('EQUILIBRIUM_PHASES')
phreeqc.AccumulateLine('Hydroxyapatite	0	0')
phreeqc.AccumulateLine('Calcite	0	0')
phreeqc.AccumulateLine('Save solution 2')
phreeqc.AccumulateLine('End')

% Simulation of the septic tank effluent
phreeqc.AccumulateLine('MIX 1')
phreeqc.AccumulateLine('1 1') %influent (solution 1), 100% flowrate
phreeqc.AccumulateLine(sprintf('2 %.16g',parm.rec)) %slag filter effluent (solution 2) at a recirculation ratio of parm.rec
phreeqc.AccumulateLine('REACTION')
phreeqc.AccumulateLine(sprintf('CO2(g) %.16g',0.001*model.CO2/12')) %CO2 from biological activity
phreeqc.AccumulateLine('Save solution 3')
phreeqc.AccumulateLine('End')
phreeqc.AccumulateLine(sprintf('Use solution 3'))
phreeqc.AccumulateLine('EQUILIBRIUM_PHASES')
phreeqc.AccumulateLine('Hydroxyapatite	0	0')
phreeqc.AccumulateLine('Calcite	0.6	0') %precipitation of calcite in the septic tank
phreeqc.AccumulateLine('End')
phreeqc.RunAccumulated();

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Data extraction

F=phreeqc.GetSelectedOutputArray();
output=[F{3,1} F{3,2} F{3,3} F{3,4} F{3,5};F{4,1} F{4,2} F{4,3} F{4,4} F{4,5}; F{6,1} F{6,2} F{6,3} F{6,4} F{6,5}];
