/*****************************************************************************
 *
 * Copyright (c) 2000 - 2012, Lawrence Livermore National Security, LLC
 * Produced at the Lawrence Livermore National Laboratory
 * LLNL-CODE-442911
 * All rights reserved.
 *
 * This file is  part of VisIt. For  details, see https://visit.llnl.gov/.  The
 * full copyright notice is contained in the file COPYRIGHT located at the root
 * of the VisIt distribution or at http://www.llnl.gov/visit/copyright.html.
 *
 * Redistribution  and  use  in  source  and  binary  forms,  with  or  without
 * modification, are permitted provided that the following conditions are met:
 *
 *  - Redistributions of  source code must  retain the above  copyright notice,
 *    this list of conditions and the disclaimer below.
 *  - Redistributions in binary form must reproduce the above copyright notice,
 *    this  list of  conditions  and  the  disclaimer (as noted below)  in  the
 *    documentation and/or other materials provided with the distribution.
 *  - Neither the name of  the LLNS/LLNL nor the names of  its contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT  HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR  IMPLIED WARRANTIES, INCLUDING,  BUT NOT  LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS FOR A PARTICULAR  PURPOSE
 * ARE  DISCLAIMED. IN  NO EVENT  SHALL LAWRENCE  LIVERMORE NATIONAL  SECURITY,
 * LLC, THE  U.S.  DEPARTMENT OF  ENERGY  OR  CONTRIBUTORS BE  LIABLE  FOR  ANY
 * DIRECT,  INDIRECT,   INCIDENTAL,   SPECIAL,   EXEMPLARY,  OR   CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT  LIMITED TO, PROCUREMENT OF  SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF  USE, DATA, OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER
 * CAUSED  AND  ON  ANY  THEORY  OF  LIABILITY,  WHETHER  IN  CONTRACT,  STRICT
 * LIABILITY, OR TORT  (INCLUDING NEGLIGENCE OR OTHERWISE)  ARISING IN ANY  WAY
 * OUT OF THE  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 *
 *****************************************************************************/

// ************************************************************************* //
//                            avtcm1visitFileFormat.C                           //
// ************************************************************************* //

#include <avtcm1visitFileFormat.h>

#include <string>

#include <vtkFloatArray.h>
#include <vtkRectilinearGrid.h>
#include <vtkStructuredGrid.h>
#include <vtkUnstructuredGrid.h>

#include <avtDatabaseMetaData.h>

#include <DBOptionsAttributes.h>
#include <Expression.h>

#include <InvalidVariableException.h>

// If you're looking for someone to blame for this code, see: Leigh Orf  <leigh.orf@wisc.edu>
// Huge thanks to Rob Sisneros (NCSA) for writing the mesh, ghost zone, and
// on-the-fly decomposition code that lets you choose as many MPI ranks
// as you wish

using     std::string;

void decompose(int *, int *);


// ****************************************************************************
//  Method: avtcm1visitFileFormat constructor
//
//  Programmer: sisneros -- generated by xml2avt
//  Creation:   Mon Aug 13 14:27:50 PST 2012
//
// ****************************************************************************

#include <DebugStream.h>

//ORF: Everything is now stored in one file (used to be hdforf.h)
//LOFS=="Leigh Orf File System" or "Lack Of File System" (Heyyyy Rockyyyy)
extern "C" {
#include <lofs-read.h>
}

    avtcm1visitFileFormat::avtcm1visitFileFormat(const char *filename) : avtMTMDFileFormat(filename)
{
    int i,nnodedirs;
    hid_t file_id,strtype;

    if ((file_id = H5Fopen (filename, H5F_ACC_RDONLY,H5P_DEFAULT)) < 0)
    {
        fprintf(stderr,"Cannot open %s, something is seriously amiss!\n",filename);
        ERROR_STOP("Cannot open hdf file");
    }
    H5LTread_dataset_string(file_id,"/topdir",topdir);
    H5LTread_dataset_int(file_id,"/ntimedirs",&ntimedirs);
    dirtimes = (double *)malloc(ntimedirs * sizeof(double));
    timedir = (char **)malloc(ntimedirs * sizeof(char *)); for (i=0; i < ntimedirs; i++) timedir[i] = (char *)(malloc(256 * sizeof(char)));
    strtype=H5Tcopy(H5T_C_S1); H5Tset_size(strtype,H5T_VARIABLE);
    H5LTread_dataset(file_id,"/timedir",strtype,timedir);
    H5LTread_dataset_int(file_id,"/nnodedirs",&nnodedirs);
    nodedir = (char **)malloc(nnodedirs * sizeof(char *)); for (i=0; i < nnodedirs; i++) nodedir[i] = (char *)(malloc(7 * sizeof(char)));
    strtype=H5Tcopy(H5T_C_S1); H5Tset_size(strtype,H5T_VARIABLE);
    H5LTread_dataset(file_id,"/nodedir",strtype,nodedir);
    H5LTread_dataset_int(file_id,"/dn",&dn);
    H5LTread_dataset_int(file_id,"/ntottimes",&ntottimes);
    alltimes = (double *)malloc(ntottimes * sizeof(double));
    H5LTread_dataset_double(file_id,"/alltimes",alltimes);
    H5LTread_dataset_double(file_id,"/dirtimes",dirtimes);
    H5LTread_dataset_int(file_id,"/nx",&nx);
    H5LTread_dataset_int(file_id,"/ny",&ny);
    H5LTread_dataset_int(file_id,"/nz",&nz);

    /* Legacy variables; using X0 X1 etc. now, but for backward
     * compatibility with older cm1hdf5 files, we are still writing
     * these to cm1hdf5 files, but also X0 X1 etc. */
    H5LTread_dataset_int(file_id,"/snx0",&snx0);
    H5LTread_dataset_int(file_id,"/snx1",&snx1);
    H5LTread_dataset_int(file_id,"/sny0",&sny0);
    H5LTread_dataset_int(file_id,"/sny1",&sny1);
    H5LTread_dataset_int(file_id,"/snz0",&snz0);
    H5LTread_dataset_int(file_id,"/snz1",&snz1);

    H5LTread_dataset_int(file_id,"/nodex",&nodex);
    H5LTread_dataset_int(file_id,"/nodey",&nodey);
    xhfull = (float *)malloc(nx * sizeof(float)); yhfull = (float *)malloc(ny * sizeof(float)); zh = (float *)malloc(nz * sizeof(float));
    H5LTread_dataset_float(file_id,"/xhfull",xhfull);
    H5LTread_dataset_float(file_id,"/yhfull",yhfull);
    H5LTread_dataset_float(file_id,"/zh",zh); for (i=0; i < nz; i++)
    H5LTread_dataset_int(file_id,"/nvars",&nvars);
    strtype=H5Tcopy(H5T_C_S1); H5Tset_size(strtype,40);
    H5LTread_dataset (file_id, "/varname", strtype, *varname);
    H5Fclose(file_id); // This is important!

    debug4 << "ORF ntimedirs = " << ntimedirs << endl;
    debug4 << "Spacing between node directories: " << dn << endl;
    debug4 << "ORF: nx = " << nx << " ny = " << ny << " nz = " << nz << " nodex = " << nodex << " nodey = " << nodey << endl;

    sizex = snx1-snx0+1;
    sizey = sny1-sny0+1;
    sizez = snz1-snz0+1;

    debug4 << endl << "Number of time steps: " << ntottimes << endl;
}

// ****************************************************************************
//  Method: avtEMSTDFileFormat::GetNTimesteps
//
//  Purpose:
//      Tells the rest of the code how many timesteps there are in this file.
//
//  Programmer: sisneros -- generated by xml2avt
//  Creation:   Mon Aug 13 14:27:50 PST 2012
//
// ****************************************************************************

    int
avtcm1visitFileFormat::GetNTimesteps(void)
{
    return ntottimes;
}


// ****************************************************************************
//  Method: avtcm1visitFileFormat::FreeUpResources
//
//  Purpose:
//      When VisIt is done focusing on a particular timestep, it asks that
//      timestep to free up any resources (memory, file descriptors) that
//      it has associated with it.  This method is the mechanism for doing
//      that.
//
//  Programmer: sisneros -- generated by xml2avt
//  Creation:   Mon Aug 13 14:27:50 PST 2012
//
// ****************************************************************************

    void
avtcm1visitFileFormat::FreeUpResources(void)
{
}

    void
avtcm1visitFileFormat::GetTimes(std::vector<double> &times)
{
    for(int i = 0; i < ntottimes; i++)
        times.push_back(alltimes[i]);
}

    void
avtcm1visitFileFormat::GetCycles(std::vector<int> &cycles)
{
    for(int i = 0; i < ntottimes; i++)
        cycles.push_back(i);
}


// ****************************************************************************
//  Method: avtcm1visitFileFormat::PopulateDatabaseMetaData
//
//  Purpose:
//      This database meta-data object is like a table of contents for the
//      file.  By populating it, you are telling the rest of VisIt what
//      information it can request from you.
//
//  Programmer: sisneros -- generated by xml2avt
//  Creation:   Mon Aug 13 14:27:50 PST 2012
//
// ****************************************************************************

    void
avtcm1visitFileFormat::PopulateDatabaseMetaData(avtDatabaseMetaData *md, int timeState)
{
    string meshname = "mesh";

    avtMeshType mt = AVT_RECTILINEAR_MESH;

    int nblocks = 1;
    int block_origin = 0;
    int spatial_dimension = 3;
    int topological_dimension = 3;
    double *extents = NULL;

    AddMeshToMetaData(md, meshname, mt, extents, nblocks, block_origin,
            spatial_dimension, topological_dimension);
    md->SetFormatCanDoDomainDecomposition(true);

    avtCentering cent = AVT_NODECENT;
    for(int i = 0; i < nvars; i++)
        AddScalarVarToMetaData(md, varname[i], meshname, cent);
}


// ****************************************************************************
//  Method: avtcm1visitFileFormat::GetMesh
//
//  Purpose:
//      Gets the mesh associated with this file.  The mesh is returned as a
//      derived type of vtkDataSet (ie vtkRectilinearGrid, vtkStructuredGrid,
//      vtkUnstructuredGrid, etc).
//
//  Arguments:
//      timestate   The index of the timestate.  If GetNTimesteps returned
//                  'N' time steps, this is guaranteed to be between 0 and N-1.
//      meshname    The name of the mesh of interest.  This can be ignored if
//                  there is only one mesh.
//
//  Programmer: sisneros -- generated by xml2avt
//  Creation:   Mon Aug 13 14:27:50 PST 2012
//
// ****************************************************************************

#include <vtkIntArray.h>
#include <vtkFieldData.h>
#include <avtGhostData.h>
#include <vtkUnsignedCharArray.h>
#include <vtkPointData.h>
#include <vtkCellData.h>
#include <vtkStreamingDemandDrivenPipeline.h>

    vtkDataSet *
avtcm1visitFileFormat::GetMesh(int timestate, int domain, const char *meshname)
{
    if (strcmp(meshname, "mesh") != 0)
    {    
        EXCEPTION1(InvalidVariableException, meshname);
    }

    int   i;   
    hid_t file_id;

    int myRank=0;
#ifdef PARALLEL
    MPI_Comm_rank(VISIT_MPI_COMM, &myRank);
#endif

    int sizes[3] = {sizex, sizey, sizez};
    int nBlks[3];
    decompose(nBlks, sizes);

    //block sizes
    int bsX = sizex/nBlks[0];
    int bsY = sizey/nBlks[1];
    int bsZ = sizez/nBlks[2];

    if(myRank >= nBlks[0]*nBlks[1]*nBlks[2])
        return NULL;

    long long x_off = myRank % nBlks[0];
    long long y_off = ((myRank-x_off)/nBlks[0])%nBlks[1];
    long long z_off = (((myRank-x_off)/nBlks[0])-y_off)/nBlks[1];

    //find the start/end of each block
    long long x_start = snx0+bsX*x_off;
    long long x_stop  = snx0+bsX*(x_off+1);
    long long y_start = sny0+bsY*y_off;
    long long y_stop  = sny0+bsY*(y_off+1);
    long long z_start = snz0+bsZ*z_off;
    long long z_stop  = snz0+bsZ*(z_off+1);

    //Create the VTK construct.
    vtkRectilinearGrid *rv = vtkRectilinearGrid::New();

    long long or_xStart = x_start;
    long long or_xStop = x_stop;
    long long or_yStart = y_start;
    long long or_yStop = y_stop;
    long long or_zStart = z_start;
    long long or_zStop = z_stop;

    //adjust to match mesh
    if(x_stop < snx1)
        x_stop++; 
    if(y_stop < sny1)
        y_stop++; 
    if(z_stop < snz1)
        z_stop++; 

    //add ghost
    if(x_start > snx0)
        x_start--;
    if(x_stop < snx1)
        x_stop++; 
    if(y_start > sny0)
        y_start--;
    if(y_stop < sny1)
        y_stop++; 
    if(z_start > snz0)
        z_start--;
    if(z_stop < snz1)
        z_stop++; 

    vtkFloatArray *x = vtkFloatArray::New();
    long long dx = x_stop-x_start;
    x->SetNumberOfTuples(dx);

    for (i = 0 ; i < dx ; i++)
        x->SetTuple1(i, xhfull[x_start+i]);

    vtkFloatArray *y = vtkFloatArray::New();
    long long dy = y_stop-y_start;

    y->SetNumberOfTuples(dy);
    for (i = 0 ; i < dy ; i++)
        y->SetTuple1(i, yhfull[y_start+i]);

    vtkFloatArray *z = vtkFloatArray::New();
    long long dz = z_stop-z_start; 

    z->SetNumberOfTuples(dz);
    for (i = 0 ; i < dz ; i++)
        z->SetTuple1(i, zh[z_start+i]);

    debug4 << "GetMesh: Domain " << domain <<
        " X range " << x_start << 
        " to " << x_stop << " Y range " << 
        y_start << " to " << y_stop << endl;

    int dims[3] = { dx, dy, dz };
    rv->SetDimensions(dims);
    rv->SetXCoordinates(x);
    rv->SetYCoordinates(y);
    rv->SetZCoordinates(z);
    x->Delete();
    y->Delete();
    z->Delete();

    x_stop--;
    y_stop--;
    z_stop--;

    int nzX = x_stop-x_start;
    int nzY = y_stop-y_start;
    int nzZ = z_stop-z_start;

    int nzones = nzX*nzY*nzZ;
    int *blanks = new int[nzones];

    for(int x = x_start, i = 0; x < x_stop; x++, i++) 
        for(int y = y_start, j = 0; y < y_stop; y++, j++) 
            for(int z = z_start, k = 0; z < z_stop; z++, k++) 
            {
                if(x < or_xStart || x >= or_xStop ||
                        y < or_yStart || y >= or_yStop ||
                        z < or_zStart || z >= or_zStop)
                    blanks[k*nzX*nzY+j*nzX+i] = 1;
                else
                    blanks[k*nzX*nzY+j*nzX+i] = 0;
            }

    unsigned char realVal = 0, ghost = 0;
    avtGhostData::AddGhostZoneType(ghost, DUPLICATED_ZONE_INTERNAL_TO_PROBLEM);
    vtkUnsignedCharArray *ghostzones = vtkUnsignedCharArray::New();
    ghostzones->SetName("avtGhostZones");
    ghostzones->Allocate(nzones);
    for(int i = 0; i < nzones; i++)
    {
        if(blanks[i])
            ghostzones->InsertNextValue(ghost);
        else
            ghostzones->InsertNextValue(realVal);
    }
    rv->GetCellData()->AddArray(ghostzones);
//    rv->SetUpdateGhostLevel(0);
    vtkStreamingDemandDrivenPipeline::SetUpdateGhostLevel(rv->GetInformation(), 0);
    ghostzones->Delete();

    delete [] blanks;

    return rv;
}


// ****************************************************************************
//  Method: avtcm1visitFileFormat::GetVar
//
//  Purpose:
//      Gets a scalar variable associated with this file.  Although VTK has
//      support for many different types, the best bet is vtkFloatArray, since
//      that is supported everywhere through VisIt.
//
//  Arguments:
//      timestate  The index of the timestate.  If GetNTimesteps returned
//                 'N' time steps, this is guaranteed to be between 0 and N-1.
//      varname    The name of the variable requested.
//
//  Programmer: sisneros -- generated by xml2avt
//  Creation:   Mon Aug 13 14:27:50 PST 2012
//
// ****************************************************************************

    vtkDataArray *
avtcm1visitFileFormat::GetVar(int timestate, int domain,  const char *varname)
{
    int   i;
    hid_t file_id;

    int myRank=0;
#ifdef PARALLEL
    MPI_Comm_rank(VISIT_MPI_COMM, &myRank);
#endif

    int sizes[3] = {sizex, sizey, sizez};
    int nBlks[3];
    decompose(nBlks, sizes);

    //block sizes
    int bsX = sizex/nBlks[0];
    int bsY = sizey/nBlks[1];
    int bsZ = sizez/nBlks[2];

    if(myRank >= nBlks[0]*nBlks[1]*nBlks[2])
        return NULL;

    long long x_off = myRank % nBlks[0];
    long long y_off = ((myRank-x_off)/nBlks[0])%nBlks[1];
    long long z_off = (((myRank-x_off)/nBlks[0])-y_off)/nBlks[1];

    long long x_start = snx0+bsX*x_off;
    long long x_stop  = snx0+bsX*(x_off+1);
    long long y_start = sny0+bsY*y_off;
    long long y_stop  = sny0+bsY*(y_off+1);
    long long z_start = snz0+bsZ*z_off;
    long long z_stop  = snz0+bsZ*(z_off+1);

    //adjust for matching mesh/ghost nodes
    if(x_start > snx0)
        x_start--;
    if(x_stop < snx1) 
        x_stop+=2;
    if(y_start > sny0)
        y_start--;
    if(y_stop < sny1) 
        y_stop+=2;
    if(z_start > snz0)
        z_start--;
    if(z_stop < snz1) 
        z_stop+=2;

    // Create the VTK construct.
    vtkFloatArray *rv = vtkFloatArray::New();
    rv->SetNumberOfComponents(1);
    rv->SetNumberOfTuples((x_stop-x_start)*(y_stop-y_start)*(z_stop-z_start));

    //read requires actual indices
    x_stop--;
    y_stop--;
    z_stop--;

    debug4 << "GetVar: Domain " << domain << " X range " << x_start << " to " << x_stop << " Y range " << y_start << " to " << y_stop << endl;

    read_hdf_mult_md((float *)rv->GetVoidPointer(0), topdir, timedir, 
            nodedir, ntimedirs, dn, dirtimes, alltimes, ntottimes, alltimes[timestate], (char *)varname, 
            x_start, y_start, x_stop, y_stop, z_start, z_stop, 
            nx, ny, nz, nodex, nodey);
    return rv;
}

void normalize(float *v, int size)
{
    float mag = 0;
    for(int i = 0; i < size; i++)
        mag += pow(v[i], 2);
    mag = sqrt(mag);
    for(int i = 0; i < size; i++)
        v[i] /= mag;
}

float distance(float *v1, float *v2, int size)
{
    float d = 0;
    for(int i = 0; i < size; i++)
        d += pow(v1[i]-v2[i],2);

    return sqrt(d);
}

void decompose(int *numBlocks, int *sizes)
{
    float ratios[3] = {sizes[0]*1./sizes[0], 
                       sizes[1]*1./sizes[0],
                       sizes[2]*1./sizes[0]};
    numBlocks[0] = numBlocks[1] = numBlocks[2] = 1;
    int stop[3] = {0, 0, 0};
    float check[3];

    float min, d;
    int index;

    int numProcs = 1;
#ifdef PARALLEL
    MPI_Comm_size(VISIT_MPI_COMM, &numProcs);
#endif
    normalize(ratios, 3);

    for(int b = 1; b < numProcs; b++)
    {
        min = -1;
        index = -1;

        for(int i = 0; i < 3; i++)
        {
            if(stop[i])
                continue;
            for(int j = 0; j < 3; j++)
                check[j] = numBlocks[j]; 
            check[i]+=1;
            normalize(check, 3);
            d = distance(check, ratios, 3);
            if(min == -1 || d < min)
            {
                min = d;
                index = i;
            }
        }
        if(index != -1)
            numBlocks[index]++;
        for(int i = 0; i < 3; i++)
        {
            numBlocks[i]++;
            if(numBlocks[0]*numBlocks[1]*numBlocks[2] > numProcs)
                stop[i] = 1;
            numBlocks[i]--;
        }
        if(stop[0] && stop[1] && stop[2])
            break;
    }
    for(int i = 0; i < 3; i++)
        while(sizes[i]%numBlocks[i] != 0)
            numBlocks[i]--;
}





// ****************************************************************************
//  Method: avtcm1visitFileFormat::GetVectorVar
//
//  Purpose:
//      Gets a vector variable associated with this file.  Although VTK has
//      support for many different types, the best bet is vtkFloatArray, since
//      that is supported everywhere through VisIt.
//
//  Arguments:
//      timestate  The index of the timestate.  If GetNTimesteps returned
//                 'N' time steps, this is guaranteed to be between 0 and N-1.
//      domain     The index of the domain.  If there are NDomains, this
//                 value is guaranteed to be between 0 and NDomains-1,
//                 regardless of block origin.
//      varname    The name of the variable requested.
//
//  Programmer: sisneros -- generated by xml2avt
//  Creation:   Mon Aug 13 14:27:50 PST 2012
//
// ****************************************************************************

    vtkDataArray *
avtcm1visitFileFormat::GetVectorVar(int timestate, int domain,const char *varname)
{
    EXCEPTION1(InvalidVariableException, varname);
}
