%% plotHVAC 
% Specifically used to plot HVAC price related curves. 
% 
% Pelec: Total HVAC electricity cost.
% PHVAC: Actual HVAC power consumption (watts)
% cost(): HECO hourly cost vector. 

function [Pelec,wattHoursConsumed] = plotHVAC(PHVAC,cost,doy,interval,div,hideFigures)
tbegin = '22-Jun-2013 00:00:00';    % Referenced and manipulated in plotHVAC and plotTemperature.
tend = '22-Jun-2013 24:00:00';      % Change the date only (not time).
costVec = [cost(doy,:),cost(doy,end)]; % Just for plotting purposes.
    if hideFigures == 0 
        figure(1)
        subplot(3,1,1)
        plotCosts(costVec,tbegin,tend,[1 1 1 60],'-b*'); ylabel('$/MWh'), xlabel('Time (hours)'), grid on, title('Electricity Price')
    end 
    SCALE = 1;   % SCALE = 1 for the 5 room example. 400x for UH

% Each of these elec_costs assumes the 5 room example: 63.6 m^2 (5000 ft^2).
% http://www.manoa.hawaii.edu/facilities/MiscDocs/status%20reports/FCA_Executive_Summary_January_2011.pdf
% Above says 4 million gross square feet over all buildings. Let's say
% that 50% of this area needs AC. Then that's 2e6/5000 = 400x elec_cost. 

    for j = 1:size(PHVAC,2)
        if hideFigures == 0, subplot(3,1,2), end % Pass actual cost here, not costVec; only want the hour markers.
        [Pelec,wattHoursConsumed] = elecPrice(PHVAC(:,j),cost,doy,SCALE,tbegin,tend,interval,div,hideFigures); hold all,
        
        multCostVec = kron(costVec(:,1:end-1)',ones(div,1)); % Used for reinterpreted Pelec 
        PelecVec = multCostVec.*PHVAC./(1e6);
        Pelec = PelecVec;
            
        if hideFigures == 0,subplot(3,1,3)
            %PelecVec = [Pelec,Pelec(end)];
            plotCosts(PelecVec,tbegin,tend,interval,'.-'); ylabel('$ for AC (hourly price)'), grid on, title('Total HVAC Electricity Price')
            hold all
        end
    end 
    if hideFigures == 0
        subplot(3,1,2), rotateticklabel(gca);
        subplot(3,1,3), rotateticklabel(gca);
    end

end 

%% elecPrice
% HVAC electricity price integrator and price finder.

function [Pelec,wattHoursConsumed] = elecPrice(PHVAC,cost,doy,SCALE,tbegin,tend,interval,div,hideFigures)
    if hideFigures == 0
        tvec = plotDate(PHVAC,tbegin,tend,interval,'.-');
        ylabel('Power (Watts)'), xlabel('Time (15 Min Intervals)'), grid on, title('Total HVAC Electric Power Consumption')
    end
    wattHoursConsumed = zeros(1,24);    % Contains integrated electricity prices for each hour itnerval. 
    for i = 0:1:23
        interval = [i,i+1];         % Beginning and ending hours to check       
        off1 = 0; off2 = 0;
        if interval(1) == 0,    off1 = 1;   end         % Off-by-one checks to make sure summing is proper.
        %if interval(2) == 24,   off2 = 0;   end 
            time = div*interval(1)+off1:1:div*interval(2)-off2;    % Find time samples to integrate over (5 min divs)
            sample = transpose(PHVAC(time));                        % Evaluate power measured at those timestamps
        wattHoursConsumed(i+1) = trapz(time.*1/div,sample);        % "Integration." Unit: Watt Hours (Wh)
        % Mind the 1/div above; this is because we aren't truly
        % integrating.
        xverts = [time(1:end-1);time(1:end-1);time(2:end);time(2:end)]; 
        yverts = [zeros(1,length(sample)-1);sample(1:end-1);sample(2:end);zeros(1,length(sample)-1)];
        hold on
        if hideFigures == 0
          %  p = patch((xverts-1)./(length(PHVAC)-1)+tvec(1),yverts,'b','LineWidth',0.1);
        end
        %i,pause
    end
    Pelec = (10^-6)*wattHoursConsumed.*cost(doy,:);  % Cost of AC at beginning of each hour.
    Pelec = Pelec.*SCALE;                     % We do NOT use costVec here since we just want the hour markers.
end 

function [tvec,t] = plotCosts(arr,strsrt,strend,step,plotstr,mode)     % plotstr is just the modifier for custom plots
    
    start = datenum(strsrt);  last = datenum(strend);
    t = linspace(start,last,length(arr));
    
    if nargin == 6 && strcmp(mode,'until')      % Special mode handle
        t = linspace(start,last,length(arr)/2);
        t = kron(t,ones(1,2));
        t = [t(2:end), t(end)];
    end 
    
    tvec = datelist(start, last, 1:step(1):12, 1:step(2):31,1:step(3):24,0:step(4):60);
    
    if nargin == 5
        stairs(t, arr, plotstr,'LineWidth',2)
    else
        stairs(t, arr,'LineWidth',2)
    end
    
        set(gca, 'xlim', [start last], 'xtick', tvec)
        datetick('keepticks', 'keeplimits'), grid on;
end 




